#!/usr/bin/perl -w
# command interpreter for Serial Addressable RGB LED PWM Driver
# http://picprojects.org.uk//zcode/
#
# Pete Griffiths, October 2007
#
# 1.1.4 code adds servo command support
#
# Disclaimer 
# All information, software and firmware is provided "as is", and without
# warranty of any kind, either express or implied.  
# In no event shall I be liable to you or any third party for any consequential,
# incidental, direct, indirect, special, punitive or other damages arising out
# of the use or inability to use this software.
#
#
# Sorry this code is all a bit of a mess as I wrote it for my own use without
# ever intending to make it public.  
#
# When I decided to release it for use with the website project it got extensively
# modified and probably should have been re-written from scratch but didn't.
#
# It's not a great example for good programming practice but hey, it does the job.
#
# ================================================================================
# Command summary
#
# define address 'address_name' = [addr1, addr2, .....]
#
# define colour 'colour_name' = Rvalue, Gvalue, Bvalue
#
# colour 'colour_name' [nofade | fade rate] to [address | 'address_name']
#
# transfer to [address | address_name]
#
# select from [subname1, subname2, ...]
#
# call 'subname'
#
# pause [ time | rand [range | base range]]  (delay in seconds)
#
# fastpause [ time | rand [range | base range]] (delay in 1/10ths seconds)
#
# repeat count
#
# next
#
# sub 'name'
#
# return
#
# print "any_text"
#
# keepalive
#
# restart
#
# stop
# 
# # comments line
#
# ================================================================================
#
#
use strict;
use Win32::SerialPort; # needs win32::api
my $version = '1.1.5';
my $VERBOSE = 0;
my $DEBUG   = 0;
my $filename = "";
my @commands;
my %subroutine;
my $baud = "9600";
my $port = 1;
my $lineNo = 0;
my %namedAddr;
my %colours;
my %positions;
my @repeatCount;
my @linePointer;
my $nested=0;
my @subreturn;
my $subcount;



if (@ARGV){
    
        my $cl_string = join (";",@ARGV) .";";
        chomp $cl_string;
              
        # Inspect command line string and set options if found
        $DEBUG   = 1 if $cl_string=~ /\/d/; # look for /d;
        $VERBOSE = 1 if $cl_string=~ /\/v/; # look for /d;
             
        $baud = $1 if $cl_string=~ /\/b:(\d+);/; # look for /i:(any digits);
        $port = $1 if $cl_string=~ /\/p:(\d+);/; # look for /p:(any digits);
                
        $filename = ($1) if $cl_string=~ /\/s:(.*?);/; # look for /s:(any characters);
        
        showUse() if $cl_string=~ /(\?|\/help)/; # look for ? or /help
}

if ($port<1 or $port >4){
        die "Comm port $port is out of range. Use 1-4 \n\n";
}
$port = "com$port";

die "No command file specified. Use chromic.exe /? for help\n\n" unless $filename;        
print "chromic RGB driver command interpreter $version\n";
print "--------------------------------------------\n";
print "Serial bit rate is : $baud\n";
print "Port is            : $port\n";
print "Command file is    : $filename\n";
print "Debug mode ON\n" if $DEBUG;
print "Verbose mode ON\n" if $VERBOSE;
print "--------------------------------------------\n";

# Configure serial port
my $Config_File = "chromic.cfg";
my $parity = "none";
my $databits = "8";
my $stopbits = "1";
my $handshake = "none";

$|=1; 

my $PortObj = new Win32::SerialPort ($port)
       || die "Can't open $port: $^E\n";    # $quiet is optional


  $PortObj->baudrate($baud);
  $PortObj->parity($parity);
  $PortObj->databits($databits);
  $PortObj->stopbits($stopbits);
  #$PortObj->handshake($handshake);
  $PortObj->buffers(6, 6);
  $PortObj->write_char_time(0);
  $PortObj->write_const_time(0);
  $PortObj->is_binary(0);
  #$PortObj->datatype('raw');
  #use open IN => ":raw", OUT => ":raw";

  $PortObj->write_settings || undef $PortObj;
  $PortObj->save($Config_File)
       || warn "Can't save $Config_File: $^E\n";
       
 $PortObj->close || die "failed to close";
  undef $PortObj; 

  my $tie_ob = tie(*PORT,'Win32::SerialPort', $Config_File)
                 or die "Can't start $Config_File\n";
 #End of serial port configuration
 

# Setup pre-defined named addresses
while (<DATA>){
         my($name, $address) = split /,/;
         $name =~ s/^\s+//;
         $name =~ s/\s+$//;
         $address =~ s/^\s+//;
         $address =~ s/\s+$//;
        $namedAddr{$name} = [$address];
}

preDefineColours();
preDefineServoPos();

# find all subroutine definitions
my $lastLineNo = 0;
my ($command, $subname) = (0,0);
{  open (IN, $filename) || die "Can't open $filename $!\n";
   my $pairs=0; 
    while (my $line = <IN>) {
        
        chomp $line;
        #next if $line =~ /^#/;
        
        $line =~ s/\s+/ /g;
        $line =~ s/^\s+//;
        $line =~ s/\s+$//;
        $line = ' ' unless $line;
        
        ($command, $subname) = split / /, $line, 2;
        if ($command eq 'sub' && $subname){
            die "Defining subroutines within another subroutine is not allowed \n" if ($pairs>0);
            die "Error: subroutine '$subname' has already been declared\n" if ($subroutine{$subname});
            $pairs++;    
            $subroutine{$subname} = $lastLineNo;
            print "Declaring subroutine '$subname'\n" if $VERBOSE;
            
        }
        if ($command eq 'return'){
            $pairs--;    
        }
        
        
        push @commands, $line;
        #print "$lastLineNo, $line\n";
        $lastLineNo++;
    }
    close IN;
    die "Found return command without a matching sub\n" if $pairs;
    print "\n\n";
}



# Lets go....
runSequence();
print "\nExiting\n\n";

# The end

sub runSequence {
    
        
    while ($lineNo < $lastLineNo){
     
            my @fields = split / /, $commands[$lineNo], 6;
            
            print $lineNo+1 .": $commands[$lineNo] \n" if ($DEBUG>0);
             
            ################################################################
            # This section skips all code between a sub and a return command
            if ($fields[0] eq 'sub'){
                $subcount++;                
            }
            if ($fields[0] eq 'return' && $subcount){
                $subcount--;
                $lineNo++;
                next;
            }
            if ($subcount){
                $lineNo++;
                next;
            } 
            ################################################################  
              
              
             if ($commands[$lineNo] =~ /^select from/){
                fselect();
             }elsif ($fields[0] eq 'call'){
                fcall(@fields);
             }elsif ($fields[0] eq 'return'){
                freturn();
             }elsif ($commands[$lineNo] =~ /^define address (.+)/){
                defAddress(@fields);
             }elsif ($commands[$lineNo] =~ /^define (colour|color) (.+)/){
                defineColours();
                
                }elsif ($commands[$lineNo] =~ /^define position (.+)/){
                definePosition();
                
             }elsif($commands[$lineNo] =~ /^transfer to (.+)/){
                transferColour(@fields);
             }elsif($commands[$lineNo] =~ /^move to (.+)/){
                transferServo(@fields);
                
             }elsif ($fields[0] eq 'keepalive'){
                keepalive();
                
             }elsif ($fields[0] eq 'colour' || $fields[0] eq 'color'){
                load(@fields);
                
             }elsif ($fields[0] eq 'servo'){
                singleServo(@fields);
                
             }elsif ($fields[0] eq 'position'){
                servoLoad(@fields);
                
             }elsif ($fields[0] eq 'pause'){
                pause(@fields);
                
             }elsif ($fields[0] eq 'fastpause'){
                fastpause(@fields)
                
             }elsif ($fields[0] eq 'repeat'){
                repeat(@fields)
                
             }elsif($fields[0] eq 'next'){
                fnext();
                
             }elsif ($fields[0] eq 'restart'){
                restart()
                
             }elsif ($fields[0] eq 'pwm'){
                pwm(@fields);
                
             }elsif ($commands[$lineNo] =~ /^print "(.+)"$/){
                print "$1\n";
             
             }elsif ($fields[0] eq 'stop'){
                print "    Exiting on stop command at line " . (1+$lineNo). "\n" if $VERBOSE;
                last;
             }elsif ($commands[$lineNo] =~ /^#|^\s|define colour .*/){
                # test for comments and blank lines
                # but do nothing.  This is needed otherwise
                # these are seen as unrecognized command
             }else{
                die "command $commands[$lineNo] not recognized at line " .(1+$lineNo) ."\n";
             }
            
            
            $lineNo++;
          
            
    }
}



sub definePosition {
    
    
        #if ($line =~ /^define colour (.+)/){
        $commands[$lineNo] =~ /^define position (.*)(=)(.*)/;
            
            my $positionName = $1;
            die "Missing '=' in define position at line " .(1+$lineNo)."\n" unless ($2);
            die "Bad or missing position values specified at line " .(1+$lineNo)."\n" unless ($3);
            
            my ($servo1, $servo2, $servo3) = split /,/, $3;
            die "Bad or missing position values specified at line " .(1+$lineNo)."\n" unless (defined $servo1 and defined $servo2 and defined $servo3);
            
            if (colourBounds($servo1, $servo2, $servo3)){
            
                $positionName =~ s/^\s+//;
                $positionName =~ s/\s+$//;
                
                $positions{$positionName}{'S1'} = $servo1;
                $positions{$positionName}{'S2'} = $servo2;
                $positions{$positionName}{'S3'} = $servo3;
                
                printf "    Defining S1=%3u, S2=%3u, S3=%3u as $positionName\n", $servo1, $servo2, $servo3 if $VERBOSE;
            }else{
                
                print "Bad servo position definition in $filename at line " . $lineNo+1 . " \n" if  $VERBOSE;
            }
    return 
}


sub defineColours {
    
    
        #if ($line =~ /^define colour (.+)/){
        $commands[$lineNo] =~ /^define (colour|color) (.*)(=)(.*)/;
            
            my $colourName = $2;
            die "Missing '=' in define colour at line " .(1+$lineNo)."\n" unless ($3);
            die "Bad or missing RGB values specified at line " .(1+$lineNo)."\n" unless ($4);
            
            my ($red, $green, $blue) = split /,/, $4;
            die "Bad or missing RGB values specified at line " .(1+$lineNo)."\n" unless (defined $red and defined $green and defined $blue);
            
            if (colourBounds($red, $green, $blue)){
            
                $colourName =~ s/^\s+//;
                $colourName =~ s/\s+$//;
                
                $colours{$colourName}{'R'} = $red;
                $colours{$colourName}{'G'} = $green;
                $colours{$colourName}{'B'} = $blue;
                
                printf "    Defining R=%3u, G=%3u, B=%3u as $colourName\n", $red, $green, $blue if $VERBOSE;
            }else{
                
                print "Bad colour definition in $filename at line " . $lineNo+1 . " \n" if  $VERBOSE;
            }
    return 
}
sub preDefineServoPos{
        
        $positions{'home'}{'S1'} = 128;
        $positions{'home'}{'S2'} = 128;
        $positions{'home'}{'S3'} = 128;
        
        $positions{'ccw'}{'S1'} = 0;
        $positions{'ccw'}{'S2'} = 0;
        $positions{'ccw'}{'S3'} = 0;
        
        $positions{'cw'}{'S1'} = 255;
        $positions{'cw'}{'S2'} = 255;
        $positions{'cw'}{'S3'} = 255;
        
        
}

sub preDefineColours {
        
        $colours{'black'}{'R'} = 0;
        $colours{'black'}{'G'} = 0;
        $colours{'black'}{'B'} = 0;
        
        $colours{'white'}{'R'} = 255;
        $colours{'white'}{'G'} = 255;
        $colours{'white'}{'B'} = 255;
        
        $colours{'red'}{'R'} = 255;
        $colours{'red'}{'G'} = 0;
        $colours{'red'}{'B'} = 0;
        
        $colours{'green'}{'R'} = 0;
        $colours{'green'}{'G'} = 255;
        $colours{'green'}{'B'} = 0;
        
        $colours{'blue'}{'R'} = 0;
        $colours{'blue'}{'G'} = 0;
        $colours{'blue'}{'B'} = 255;
        
        $colours{'magenta'}{'R'} = 255;
        $colours{'magenta'}{'G'} = 0;
        $colours{'magenta'}{'B'} = 255;
        
        $colours{'cyan'}{'R'} = 0;
        $colours{'cyan'}{'G'} = 255;
        $colours{'cyan'}{'B'} = 255;
        
        $colours{'yellow'}{'R'} = 255;
        $colours{'yellow'}{'G'} = 255;
        $colours{'yellow'}{'B'} = 0;
        
}


sub colourBounds {
            
    foreach (@_) {
           
        s/^\s+//;
        s/\s+$//;
        die "Bad definition at line ". (1+$lineNo)."\n" unless (/^\d+$/); # is it numeric?
        return 0 if ($_ < 0 || $_ > 255);
    }
    
    return 1;
}
sub namedAddr {
  
    my $address = shift @_;
    # strip spaces from start and end of string
    $address =~ s/^\s+//;
    $address =~ s/\s+$//;
    return $address if ($address =~ /^\d+$/);

    
    ($namedAddr{$address}) ? return $namedAddr{$address} : die "Named address $address not defined at line " . ($lineNo+1) ." \n";
    
    }
    
sub sendColour {
  
  my ($addr, $red, $grn, $blue, $fade) = @_;
  
  my $ptd = 0xFF;
  my $checksum = ((($ptd + $addr + $red + $grn + $blue + $fade )^255 )+1) & 255;
  my $data = chr($ptd) . chr($addr) .chr($red) .chr($grn) .chr($blue).chr($fade).chr($checksum);
   
  printPacket ($ptd,  $addr, $red, $grn, $blue, $fade, $checksum) if $DEBUG;
  syswrite PORT, $data unless $DEBUG == 2;

}

sub sendCommand {
  
  my ($addr, $command, $d1 ) = @_;
  my $d2=1;
  my $d3=0;
  my $ptd = 0xFE;
  my $checksum = ((($ptd + $addr + $command + $d1 + $d2 + $d3 )^255 )+1) & 255;
  my $data = chr($ptd) . chr($addr) .chr($command) .chr($d1) .chr($d2) .chr($d3) . chr($checksum);
   
  printPacket ($ptd, $addr, $command, $d1,  $d2, $d3, $checksum) if $DEBUG;
  syswrite PORT, $data unless $DEBUG == 2;

}


sub printPacket {
        
        print "    Sending-> command:$_[0] | address:$_[1] | data: @_[2..5] | chksum:$_[6]\n";        
        };

sub showUse {
  
            print "\n\n";
            print "chromic RGB driver command interpreter\n";
            print "Version: $version\n\n";
            print "\/s:filename    Command file\n\n";
            print "\/b:bitrate     Serial bit rate [ 1200 | 2400 | 9600 | 19200 | 38400 ]\n";
            print "\/p:comport     Serial COM port [ 1 | 2 | 3 | 4]\n\n";
            print "\/d             Debug mode on \n\n";
            print "\/v             Verbose mode on\n";
            print "\n\n http:\\\\picprojects.org.uk\\projects\\zcode\\\n";
            die "\n";
            return;
}
  
  
sub fcall{

        my @fields = @_;                
        push @subreturn, $lineNo;
        print "    Calling $fields[1] \n" if $VERBOSE;
        die "\nSubroutine $fields[1] not defined at line " . ($lineNo+1) . "\n" unless exists $subroutine{$fields[1]};
        $lineNo = $subroutine{$fields[1]};
           
}

sub freturn{

        $lineNo = pop @subreturn;
        print "    Return from call \n" if $VERBOSE;
                
}


sub fselect{
        
        $commands[$lineNo] =~ /^select from (.+)/;
        die "No subroutines in 'select from' command at line " . (1+$lineNo) . "\n" unless ($1);
        my @choices = split /,/, $1;
        my $range = @choices;
        
        my $select = int rand $range;
        my $sub = $choices[$select];
        $sub =~ s/^\s+//;
        $sub =~ s/\s+$//;
        push @subreturn, $lineNo;
        print "    Select is calling $sub \n" if $VERBOSE;
        die "\nSubroutine $sub not defined at line " . ($lineNo+1) . "\n" unless exists $subroutine{$sub};
        $lineNo = $subroutine{$sub};
      
}


sub defAddress{
        #1      2       3      4 5 
        #define address 'name' = 'address'
       my @fields = @_;      
       my $name = $fields[2];
        die "Missing '=' in define address at line ". (1+$lineNo). "\n" unless ($fields[3] eq '=');
       $commands[$lineNo] =~ /^define address .+ = (.+)/;
             
       $namedAddr{$name}=  [split /,/, $1];
       
       my $ref = $namedAddr{$name};
       
       foreach (@$ref){
              s/^\s+//;
              s/\s+$//;
              die "Address not numeric at line " . (1+$lineNo)."\n" unless (/^\d+$/);
              die "\nBad address value $_ at line " .($lineNo+1)."\n" if ($_ < 0 || $_ > 255);
       }
}
            
sub transferColour{
        
        # transfer address 'addr'
        my @fields = @_;      
        my $address = $fields[2];
        my $refaddress = namedAddr($address);
        
               
        if (ref($refaddress) eq 'ARRAY'){
                
                foreach (@$refaddress){
                                               
                        my $address = $_;
                        # strip leading and trailing spaces
                        $address =~ s/^\s+//;
                        $address =~ s/\s+$//;
                        die "\nBad address value $address at line " .($lineNo+1)."\n" if ($address < 0 || $address > 255);
                     
                        sendCommand ($address, 1, 0);
                }
        }else{
                die "\nBad address value $address at line " .($lineNo+1)."\n" if ($refaddress < 0 || $refaddress > 255);
                 sendCommand ($address, 1, 0);
                }
}


sub transferServo{
        
        # transfer address 'addr'
        my @fields = @_;      
        my $address = $fields[2];
        my $refaddress = namedAddr($address);
        
               
        if (ref($refaddress) eq 'ARRAY'){
                
                foreach (@$refaddress){
                                               
                        my $address = $_;
                        # strip leading and trailing spaces
                        $address =~ s/^\s+//;
                        $address =~ s/\s+$//;
                        die "\nBad address value $address at line " .($lineNo+1)."\n" if ($address < 0 || $address > 255);
                     
                        sendCommand ($address, 1, 0);
                }
        }else{
                die "\nBad address value $address at line " .($lineNo+1)."\n" if ($refaddress < 0 || $refaddress > 255);
                 sendCommand ($address, 7, 0);
                }
}



sub singleServo{
        
        # servo 'n' position 128 to 0
        
        my @fields = @_;      
        my $address = $fields[5];
        my $servo = $fields[1];
        my $position = $fields[3];
        
        
        
        die "syntax error in servo command at line" . ($lineNo+1) ."\n" unless (defined $address && defined $servo && defined $position);
        die "Value is not numeric in servo command at line ". (1+$lineNo). "\n" unless ($servo =~ /^\d+$/);
        die "Value is not numeric in servo command at line ". (1+$lineNo). "\n" unless ($position =~ /^\d+$/);
        die "servo value out of range at line " . ($lineNo+1) ."\n" if ($servo < 1 || $servo >3);
        die "servo position value out of range at line " . ($lineNo+1) ."\n" if ($position < 0 || $position >255);
        my $refaddress = namedAddr($address);        
        my $command = 7 + $servo; # Servo range 1-3
               
        if (ref($refaddress) eq 'ARRAY'){
                
                foreach (@$refaddress){
                                               
                        my $address = $_;
                        # strip leading and trailing spaces
                        $address =~ s/^\s+//;
                        $address =~ s/\s+$//;
                        die "\nBad address value $address at line " .($lineNo+1)."\n" if ($address < 0 || $address > 255);
                     
                        sendCommand ($address, $command, $position);
                }
        }else{
                die "\nBad address value $address at line " .($lineNo+1)."\n" if ($refaddress < 0 || $refaddress > 255);
                 sendCommand ($address, $command, $position);
                }
        print "    Servo $servo set to position $position at address $address\n" if $VERBOSE;
}




sub pwm {
        
        my @fields = @_;      
        my $address = $fields[3];
        
        my $state;
        die "\nSyntax error at line " .($lineNo+1)."\n" unless ($fields[2] eq 'to');
        if ($fields[1] eq 'enable'){
                $state=1;
        }elsif ($fields[1] eq 'disable'){
                $state=0;
        }else{
                die "\nSyntax error at line " .($lineNo+1)."\n";
        }
        
        die "\nMissing address value at line " .($lineNo+1)."\n" unless ($fields[3]);
        
        my $refaddress = namedAddr($address);
                       
        if (ref($refaddress) eq 'ARRAY'){
                
                foreach (@$refaddress){
                                               
                        my $address = $_;
                        # strip leading and trailing spaces
                        $address =~ s/^\s+//;
                        $address =~ s/\s+$//;
                        die "\nBad address value $address at line " .($lineNo+1)."\n" if ($address < 0 || $address > 255);
                     
                        sendCommand ($address, 5, $state);
                }
        }else{
                die "\nBad address value $address at line " .($lineNo+1)."\n" if ($refaddress < 0 || $refaddress > 255);
                 sendCommand ($address, 5, $state);
                }
        
        
}


        
sub keepalive{
                             
        sendCommand (255, 0, 0);
}

sub load{
        #   0     1      2       3      4         5
        # colour 'name'  rate    'rate' to        'address'
        # colour 'name'  nofade' to     'address'
        my @fields = @_;      
        my $colour   = $fields[1];
        my $address;
        my $faderate;
        
        if ($fields[2] eq "nofade"){
                $faderate = 0;
                $address = $fields[4];
        }elsif($fields[2] eq "fade"){
                $faderate = $fields[3];
                $address = $fields[5];
        }else{
                
                die "Syntax error in colour command at line ". (1+$lineNo)."\n";
                
        }
        die "Can't find expected fields in colour command at line " . ($lineNo+1) . ".  Check syntax \n" unless defined $address;              
        die "faderate $faderate is out of range at line " . ($lineNo+1) . "\n" if ($faderate < 0 || $faderate > 255);

                unless (defined $colours{$colour}){
        
            die "\nColour name $colour not defined in $filename at line " . ($lineNo+1) . "\n";
        }
                
        my $refaddress = namedAddr($address);
        
        if (ref($refaddress) eq 'ARRAY'){
                
                foreach (@$refaddress){
                                               
                        my $address = $_;
                        # strip leading and trailing spaces
                        $address =~ s/^\s+//;
                        $address =~ s/\s+$//;
                        die "\nBad address value $address at line " .($lineNo+1)."\n" if ($address < 0 || $address > 255);
                                     
                        sendColour($address,
                                      $colours{$colour}->{'R'},
                                      $colours{$colour}->{'G'},
                                      $colours{$colour}->{'B'},
                                      $faderate);
                }
        }else{
                die "\nBad address value $address at line " .($lineNo+1)."\n" if ($refaddress < 0 || $refaddress > 255);
                
                sendColour($refaddress,
                                $colours{$colour}->{'R'},
                                $colours{$colour}->{'G'},
                                $colours{$colour}->{'B'},
                                $faderate);
                }
                
        }
 
sub servoLoad{
        #   0     1      2       3      4         5
        # colour 'name'  rate    'rate' to        'address'
        # colour 'name'  nofade' to     'address'
        my @fields = @_;      
        my $positionName   = $fields[1];
        my $address;
        my $slewrate;
        
        if ($fields[2] eq "noslew"){
                $slewrate = 0;
                $address = $fields[4];
        }elsif($fields[2] eq "slew"){
                $slewrate = $fields[3];
                $address = $fields[5];
        }else{
                
                die "Syntax error in servo command at line ". (1+$lineNo)."\n";
                
        }
        die "Can't find expected fields in servo command at line " . ($lineNo+1) . ".  Check syntax \n" unless defined $address;              
        die "slewrate $slewrate is out of range at line " . ($lineNo+1) . "\n" if ($slewrate < 0 || $slewrate > 255);

                unless (defined $positions{$positionName}){
        
            die "\nServo position name $positionName not defined in $filename at line " . ($lineNo+1) . "\n";
        }
                
        my $refaddress = namedAddr($address);
        
        if (ref($refaddress) eq 'ARRAY'){
                
                foreach (@$refaddress){
                                               
                        my $address = $_;
                        # strip leading and trailing spaces
                        $address =~ s/^\s+//;
                        $address =~ s/\s+$//;
                        die "\nBad address value $address at line " .($lineNo+1)."\n" if ($address < 0 || $address > 255);
                                     
                        sendColour($address,
                                      $positions{$positionName}->{'S1'},
                                      $positions{$positionName}->{'S2'},
                                      $positions{$positionName}->{'S3'},
                                      $slewrate);
                }
        }else{
                die "\nBad address value $address at line " .($lineNo+1)."\n" if ($refaddress < 0 || $refaddress > 255);
                
                sendColour($refaddress,
                                      $positions{$positionName}->{'S1'},
                                      $positions{$positionName}->{'S2'},
                                      $positions{$positionName}->{'S3'},
                                      $slewrate);
                }
                
        }






sub pause{
        my @fields = @_;      
        
        die "No time value specified for pause command at line "  . (1+$lineNo) . "\n" unless $fields[1];
        
        
        my $base;
        my $range;
        my $seconds = $fields[1];
                       
        if ($seconds eq "rand"){
                 
                die "No rand time value specified for pause command at line "  . (1+$lineNo) . "\n" unless $fields[2];
                $base = $fields[2];
                die "Value is not numeric in pause command at line ". (1+$lineNo). "\n" unless ($fields[2] =~ /^\d+$/);
                
                if ($fields[3]){
                        die "Value is not numeric in pause command at line ". (1+$lineNo). "\n" unless ($fields[3] =~ /^\d+$/);
                        $range = 1 + $fields[3];
                                
                }else{
                        $range = $fields[2];
                        $base = 1;
                }
                
                $seconds = int $base + (rand $range);
        }else{
                
        die "Value is not numeric in pause command at line ". (1+$lineNo). "\n" unless ($fields[1] =~ /^\d+$/);
        }
        
        print "    pausing for $seconds seconds\n";
        
        my $pauseCnt = $seconds;
        while ($pauseCnt){
                Win32::Sleep(1000);
                sendCommand (255,0,0) unless ($pauseCnt % 120); # keep alive ~120sec
                $pauseCnt--;
                
                
        }
}


sub fastpause{
        my @fields = @_;      
        
        die "No time value specified for fastpause command at line "  . (1+$lineNo) . "\n" unless $fields[1];
        
        
        my $base;
        my $range;
        my $seconds = $fields[1];
                       
        if ($seconds eq "rand"){
                 
                die "No rand time value specified for fastpause command at line "  . (1+$lineNo) . "\n" unless $fields[2];
                $base = $fields[2];
                die "Value is not numeric in pause command at line ". (1+$lineNo). "\n" unless ($fields[2] =~ /^\d+$/);
                
                if ($fields[3]){
                        die "Value is not numeric in fastpause command at line ". (1+$lineNo). "\n" unless ($fields[3] =~ /^\d+$/);
                        $range = 1 + $fields[3];
                                
                }else{
                        $range = $fields[2];
                        $base = 1;
                }
                
                $seconds = int $base + (rand $range);
        }else{
                
        die "Value is not numeric in fastpause command at line ". (1+$lineNo). "\n" unless ($fields[1] =~ /^\d+$/);
        }
        
        print "    fast pausing for $seconds/10ths second\n";
        
        my $pauseCnt = $seconds;
        while ($pauseCnt){
                Win32::Sleep(100);
                sendCommand (255,0,0) unless ($pauseCnt % 120); # keep alive ~120sec
                $pauseCnt--;
                
                
        }
}




sub repeat{
        my @fields = @_;
        die "repeat count < 1 found at line " . (1+$lineNo) . "\n" if ($fields[1] < 1);
        $nested++;
        $repeatCount[$nested] = $fields[1];
        $linePointer[$nested] = $lineNo;
        print "    Start loop # $nested\n" if $VERBOSE;
        }
        
sub fnext{
        
        $repeatCount[$nested]--;
        die "Found next command before any repeat at line " . (1+$lineNo) . "\n" if ($nested == 0);
        
        if ($repeatCount[$nested]){
            $lineNo = $linePointer[$nested];
            print "    repeat loop # $nested \n" if $VERBOSE;
        }else{
            
            print "    end loop # $nested\n" if $VERBOSE;
            $nested--;
        }
}
      


sub restart{
        $nested   = 0;
        $lineNo   = 0;
        $subcount = 0;
        print "    Restarting from beginning of command file\n" if ($VERBOSE | $DEBUG);
}

__DATA__
gr0, 128
gr1, 129
gr2, 130
gr3, 131
gr4, 132
gr5, 133
gr6, 134
gr7, 135
gc0, 144
gc1, 145
gc2, 146
gc3, 147
gc4, 148
gc5, 149
gc6, 150
gc7, 151
gc8, 152
gc9, 153
gc10,154
gc11,155
gc12,156
gc13,157
gc14,158
gc15,159
broadcast, 255
all, 255
