;-----------------------------------------------------
; Software implemented drawing processor
;-----------------------------------------------------
;
; Bit 7 == 0 instruction is single byte
; Bit 7 == 1 instruction is 2 byte 
; 
; command instrution bit field decoding
;
; command byte  [7 6 5 4 3 2 1 0]
;                0 0 c c c c c c     : single byte instruction
;                0 1 c c c r r r     : single byte instruction operating on register
;
;		         : 2 byte command with RX,RY,RZ literal data value
;                1 0 0 c c c c X2    : instruction with bit0 = X2
; 	   X X Y Y Y Z Z Z     : X1 X0 Y2 Y1 Y0 Z2 Z1 Z0	
;
;		         : 2 byte command operates on register
;	   1 0 1 c c r r r     : instruction , rrr is destination for inst.data
;	   k k k k k k k k     : data
;		
;		         : 2 byte command modifies Instruction Pointer
;                1 1 0 p p p p p     ; goto p4 p3 p2 p1 p0 are high byte of address
;  	   p p p p p p p p     ; p7 p6 p5 p4 p3 p2 p1 p0 are low byte of address
;
;                1 1 1 p p p p p     ; call p4 p3 p2 p1 p0 are high byte of address, return address is pushed on stack
;  	   p p p p p p p p     ; p7 p6 p5 p4 p3 p2 p1 p0 are low byte of address
;
;
; -----------------------------------------------------
inst.load
	call	inst.fetch
	clrf          PCLATH        ; ensure PCLATH is set to program memory 2K bank page 0
	movwf	instruction

	btfsc	instruction,7	; if command bit 7 set, do 2 word commands
	goto	inst.2byte

inst.1byte
	btfsc	instruction, 6
	goto	inst.1byte.register
	
	; decode and execute 1byte instructions

	; Use a lookup table to select 1 byte non-register instruction operation.
	; This makes decode operation take same number of instruction cycles for
	; any instruction opcode so there is no overhead for instructions at the 
	; end of the list.
	movlw	HIGH inst.1byte.base
	movwf	PCLATH
	movlw	LOW inst.1byte.base
	addwf	instruction,W
	skpnc
	incf	PCLATH,F
	movwf	PCL
inst.1byte.base
	goto	inst.load		; NOP - 0x00
	goto	inst.skpz		; SKIPZ - 0x01
	goto	inst.skpnz		; SKIPNZ -0x02
	goto	inst.show		; SHOW -0x03
	goto	inst.clr		; MCLR -0x04
	goto	inst.set		; MSET -0x05
	goto	inst.inv		; MINV -0x06
	goto	inst.vox		; PUT -0x07
	;--------------------------------------------------------------
	; Bits 1,0 select the RX, RY, RZ register so they need to 
	; decode such that low order 3 bits start at 000, ending at 010
	goto	inst.decxyz		; DECX -0x08 00001000
	goto	inst.decxyz		; DECY -0x09 00001001
	goto	inst.decxyz		; DECZ -0x0A 00001010
	;--------------------------------------------------------------
	goto	inst.return		; RET -0x0B  00001011
	;--------------------------------------------------------------
	; Bits 1,0 select the RX, RY, RZ register so they need to 
	; decode such that low order 3 bits start at 000, ending at 010
	goto	inst.incxyz		; INCX -0x0C 00001100
	goto	inst.incxyz		; INCY -0x0D 00001101
	goto	inst.incxyz		; INCZ -0x0E 00001110
	;--------------------------------------------------------------
	goto	inst.clrall		; CLRALL -0x0F
	goto	inst.setall		; SETALL -0x10
	goto	inst.invall		; INVALL -0x11
	goto	inst.pushxyz		; PUSHXYZ -0x12
	goto	inst.pullxyz		; PULLXYZ -0x13
	goto	inst.shiftz.out             ; SHZI -0x14
	goto	inst.shiftz.in              ; SHZO -0x15
	goto	inst.shifty.up              ; SHYU -0x16
	goto	inst.shifty.down	; SHYD -0x17
	goto	inst.shiftx.left	; SHXL -0x18
	goto	inst.shiftx.right	; SHXR -0x19
	goto	inst.lx	              ; LINEX -0x1A
	goto	inst.ly    	              ; LINEY -0x1B
	goto	inst.lz    	              ; LINEZ -0x1C
	goto	inst.plx	              ; PLANEX -0x1D
	goto	inst.ply    	              ; PLANEY -0x1E
	goto	inst.plz    	              ; PLANEZ -0x1F			
	goto	inst.test.voxel             ; TSTvox -0x20
	goto	inst.rand.seed              ; RNDSEED -0x21
	goto	inst.timerskip              ; SKIPTOUT -0x22
	goto	inst.rotatex                ; ROTATEX -0x23
	goto	inst.xor		; XOR - 0x24
	goto	inst.add		; ADD - 0x25
	goto	inst.sub		; SUB - 0x26
	goto	inst.and		; AND - 0x27
	goto	inst.or		; OR - 0x28
	goto	inst.skpc		; SKPC - 0x29
	goto	inst.skpnc		; SKPNC - 0x2A
	goto	inst.tstsw		; TXTSW - 0x2B
	goto	inst.drop		; DROP - 0x2C
	goto	inst.swap		; SWAP - 0x2D
	goto	inst.dup		; DUP - 0x2E
	goto	inst.not		; NOT - 0x2F
	goto	inst.over		; OVER - 0x30
	goto	inst.rot		; ROT - 0x31
	goto	inst.tstz		; TSTZ - 0x32
	;------------------------------------------------------------

inst.1byte.register
;                0 1 c c c r r r     : instruction / register

	; Unpack register from instruction opcode, 
	; then use lookup table to select instruction operation
	call	unpack.register
	movlw	HIGH inst.1byte.reg.base
	movwf	PCLATH
	movlw	LOW inst.1byte.reg.base
	addwf	instruction,W
	skpnc
	incf	PCLATH,F
	movwf	PCL

inst.1byte.reg.base
	goto          inst.decr     ; 01000rrr
	goto	inst.decrsz	; 01001rrr
	goto	inst.incr 	; 01010rrr
	goto	inst.incrsz 	; 01011rrr
	goto	inst.pushr 	; 01100rrr
	goto	inst.pullr 	; 01101rrr
	goto	inst.chyr 	; 01110rrr
	goto	inst.chzr 	; 01111rrr
	;-----------------------------------------------------------

	; decode and excute 2 byte instruction
inst.2byte
			 	; 1 0 c c c c c c
	call	inst.fetch		; read data byte of 2-byte instruction
	clrf          PCLATH                      ; ensure PCLATH is set to 2K bank page 0
	movwf	inst.data
	btfsc	instruction,6 
	goto	inst.type.addr	; 1 1 p p p p p p 
	btfsc	instruction,5
	goto	inst.type.regdata	; 1 0 1 c c c c c
	;-----------------------------------------------------------

	; Instructions operating on RX,RY, RZ.
	; or byte data for other operations
	rrf	instruction,W
	andlw	0x0F
	
	skpz
	goto	$+3
	call	unpack.xyz
	goto	inst.load                   ;0 LDXYZ X,Y,Z
	
	addlw	-.1
	skpz
	goto	$+4
	call	unpack.xyz
inst.vox	call	voxel
	goto	inst.load
	
	addlw	-.1
	skpz
	goto	$+3
	call	draw.line		;2 LINE xdir,ydir,zdir,length
	goto	inst.load
	        
	addlw	-.1
	skpz
	goto	$+3
	bsf	voxel.func, Fcharyz
	goto	inst.charyz	              ;3 CHY CharVal
		
	addlw	-.1
	skpz
	goto	$+3
	bcf	voxel.func, Fcharyz
	goto	inst.charyz	              ;4 CHZ CharVal


	addlw	-.1
	skpnz
	goto	inst.timerinit              ;5 LDTMR Seconds

	addlw	-.1
	skpnz
	goto	inst.timer.addtrnd          ;6 ADDTRND MaxRand


	addlw	-.1
	skpnz
	goto	inst.sync	             ;7 SYNCEXT SyncTimerout


;-----------------------------------------
; Draw lines
inst.rotatex	call	rotate.x
	goto	inst.load


;-----------------------------------------
; Draw lines
;
inst.lx	call	save.xyz
	call	position.xyz
	call	voxel.line.x
	call	restore.xyz
	goto	inst.load


inst.ly	call          save.xyz
	clrf	vox.y
	call	position.xyz
              call	voxel.line.y
              call          restore.xyz
	goto	inst.load


inst.lz	call          save.xyz
	clrf	vox.z
	call	position.xyz
              call	voxel.line.z
	call          restore.xyz
	goto	inst.load

;-----------------------------------------
; Draw planes

inst.plx	call	save.xyz
              call	voxel.plane.x
              call          restore.xyz
	goto	inst.load


inst.ply	call          save.xyz
              call	voxel.plane.y
	call          restore.xyz
	goto	inst.load


inst.plz	call          save.xyz
	call	voxel.plane.z
	call          restore.xyz
              goto	inst.load



;-----------------------------------------
; Test voxel State
inst.test.voxel

              call          position.xyz
              bcf           flag.bit, Fzero
              andwf         INDF,W
              skpnz
              bsf           flag.bit, Fzero
              goto          inst.load

;----------------------------------------
; Test switch state

inst.tstsw	bcf	flag.bit, Fzero
	btfss	controlPort, bit.switch
	bsf	flag.bit, Fzero
	goto	inst.load


;----------------------------------------
; command type address
; [   1   1   T pc12 pc11 pc10  pc9  pc8 ]
; [ pc7 pc6 pc5 pc4  pc3   pc2  pc1  pc0 ]
;
; T=0 command is JUMP
; T=1 command is JSR,
;     save return address on stack
;     save loop counter address and count on stack

inst.type.addr	
	btfsc	instruction,5
	call	push.address	
	movfw	instruction
	andlw	0x1F
	movwf	inst.ptr.Hi
	movfw	inst.data
	movwf	inst.ptr.Lo
	goto	inst.load
inst.return
	call	pull.address
	goto	inst.load


;----------------------------------------
; 2 Byte Register instructions
; 
; 1 0 1 c c r r r     : 2 byte instruction , rrr is destination for inst.data
;		; only 2 command decode bits
		; so limited to 4 instructions 
;
inst.type.regdata

	call	unpack.register
	bcf	instruction,2
	movfw	instruction
	
	skpnz                       ;0000 0000
	goto	inst.load.reg
	
	addlw	-.1	;0000 0001
	skpnz
	goto	inst.load.rand
	
	addlw	-.1	;0000 0010
	skpnz
	goto	inst.compare
	

	goto	fault.command2


inst.load.reg
	movfw	inst.data
	movwf	INDF
	goto	inst.load
	
inst.load.rand
              movfw         inst.data
              call          rand.num
              movwf         INDF
              goto          inst.load
              
inst.compare
	bsf           flag.bit, Fzero
              movfw         inst.data
	xorwf	INDF,W
              skpz
              bcf           flag.bit, Fzero
              goto          inst.load

;----------------------------------------
; Initiate transfer of drawing buffer to the display buffer. This
; code is blocking and involves the following steps:
; 1. Wait until current hold time as expired,
; 2. Set the copy buffer flag
; 3. Sets new hold time
; 4. Waits for buffer to transfer (handled by the display driver)
;    before returning to instruction execution

inst.show	btfss	flag.bit, Fhold.tout	; ensure previous timer has timed out
	goto	$-1
	bcf	flag.bit, Fhold.tout	; clear time out flag bit
	bsf	flag.bit, Fcopy.buffer	; set buffer copy flag to ready
	movfw	holdtime
	movwf	hold.timer		; reload the timer with new delay
	btfsc	flag.bit, Fcopy.buffer	; wait for buffer flag to clear before exiting
	goto	$-1		; otherwise we may alter buffer before it's copied
	goto	inst.load	
	
;----------------------------------------
; Sync instruction
; waits for high to low transition on the Switch input 
; before continuing instruction execution

inst.sync	
	movlw	.100
	movwf	tick
	movfw	inst.data
	movwf	seconds

	bcf	flag.bit, Fzero
	
	movf	inst.data,F	; if argument to SYNCEXT was 0 then
	skpnz		; we never want the SYNCEXT to timeout
	bsf	seconds,7	; so we force the seconds counter to 128
	movf	seconds,F	; then test seconds register for zero (timeout)
	skpnz		; if timer still running wait for SYNC
	goto	inst.load	; otherwise terminate SYNCEXT and load next instruction

	btfss	controlPort, bit.switch
	goto	$-7	; wait for input to go high before starting

	
	movf	inst.data,F	; if argument to SYNCEXT was 0 then
	skpnz		; we never want the SYNCEXT to timeout
	bsf	seconds,7	; so we force the seconds counter to 128
	movf	seconds,F	; then test seconds register for zero (timeout)
	skpnz		; if timer still running wait for SYNC
	goto	inst.load	; otherwise terminate SYNCEXT and load next instruction

	btfsc	controlPort, bit.switch
	goto	$-7	; wait for input to go low before continuing

	bsf	flag.bit, Fzero
	goto	inst.load

	

;----------------------------------------
; Seed Random number generator with non-zero value
inst.rand.seed 
              call          rand.reseed
              goto          inst.load


;-----------------------------------------------
; Set voxel modify operation flag
inst.clr
	movlw	1<<Fvoxel.clr
	movwf	voxel.func
	goto	inst.load
inst.inv
	movlw	1<<Fvoxel.inv
	movwf	voxel.func
	goto	inst.load
inst.set
	movlw	1<<Fvoxel.set
	movwf	voxel.func
	goto	inst.load


;----------------------------------------
; Clear, Set, Invert all voxels in buffer
;
inst.clrall	clrw
	call	voxel.all.clr
	goto	inst.load

inst.setall
	call	voxel.all.set
	goto	inst.load

inst.invall
	call	voxel.all.inv
	goto	inst.load


;----------------------------------------
; Decrement xyz register
inst.decxyz
	movfw	instruction
	andlw	0x03
	addlw	vox.x
	movwf	FSR
	movlw	.4
	movf	INDF,F
	skpz
	decf	INDF,W
	movwf	INDF
	goto	test.zero
	goto	inst.load

inst.incxyz
	movfw	instruction
	andlw	0x03
	addlw	vox.x
	movwf	FSR
	incf	INDF,F
	movf	INDF,W
	xorlw	.5
	skpnz
	clrf	INDF

test.zero	bsf	flag.bit, Fzero
	movf	INDF,F
	skpz
	bcf	flag.bit, Fzero
	goto	inst.load


;----------------------------------------
; Decrement register skip zero

inst.decrsz
	decf	INDF,F
	skpz
	goto	inst.load
	goto	inst.skip

inst.incrsz
	incf	INDF,F
	skpz
	goto	inst.load
	goto	inst.skip

inst.decr
	bsf	flag.bit, Fzero
	decfsz	INDF,F
	bcf	flag.bit, Fzero
	goto	inst.load

inst.incr
	bsf	flag.bit, Fzero
	incfsz	INDF,F
	bcf	flag.bit, Fzero
	goto	inst.load




;----------------------------------------
; Skip instruction
inst.skpz	btfss	flag.bit, Fzero
	goto	inst.load
	goto	inst.skip

inst.skpnz	btfss	flag.bit, Fzero
	goto	inst.skip
	goto	inst.load

inst.skpc	btfss	flag.bit, Fcarry
	goto	inst.load
	goto	inst.skip

inst.skpnc	btfss	flag.bit, Fcarry
	goto	inst.skip
	goto	inst.load


;----------------------------------------
; Skip instruction
inst.skip
	call	inst.fetch
	clrf          PCLATH        ; ensure PCLATH is set to 2K bank page 0
	andlw	0x80	; test if 2word command
	skpz
	call	inst.fetch
	clrf          PCLATH        ; ensure PCLATH is set to 2K bank page 0
	goto	inst.load

;----------------------------------------
; Write Character from value in Register
inst.chyr
	movfw	INDF
	movwf	inst.data
	bsf	voxel.func, Fcharyz
	goto	inst.charyz
	
inst.chzr
	movfw	INDF
	movwf	inst.data
	bcf	voxel.func, Fcharyz
	goto	inst.charyz

;----------------------------------------
; TMR instructions

; Load timer
inst.timerinit
	movlw	.100
	movwf	tick
	movfw	inst.data
	movwf	seconds
	goto	inst.load

inst.timer.addtrnd
              movlw	.100
	movwf	tick
              movfw         inst.data
              call          rand.num
              addwf         seconds,F	; add random value to current TMR
              goto          inst.load
       
; Test timer
inst.timerskip
	movf	seconds,F	; test seconds register
	skpz		; skip next if seconds == 0
	goto	inst.load	; else continue with next instruction
	goto	inst.skip	; skip next instruction

;----------------------------------------
; Shift Z
inst.shiftz.in
	call	buffer.shift.z.in
	goto	inst.load

inst.shiftz.out
	call	buffer.shift.z.out
	goto	inst.load


;----------------------------------------
; Shift Y
inst.shifty.up
	call	buffer.shift.y.up
	goto	inst.load

inst.shifty.down
	call	buffer.shift.y.down
	goto	inst.load

;----------------------------------------
; Shift X
inst.shiftx.left
	call	buffer.shift.x.left
	goto	inst.load

inst.shiftx.right
	call	buffer.shift.x.right
	goto	inst.load

;------------------------------------------
; Write character to buffer
; Character table has definitions for ASCII characters from 0x20 [space] to 0x4F [?

inst.charyz
	call	save.xyz

	clrf	char.ptr.Hi	
	movlw	0x20		; ASCII data starts at 0x20
	subwf	inst.data,W		; subtract to get 0 index for lookup
	andlw	0x3F		; mask off unwanted high order bits
	movwf	char.ptr.Lo
		
	clrc			; multiply by five to get base index
	rlf	char.ptr.Lo,F		; offset into character data table
	rlf	char.ptr.Lo,F
	addwf	char.ptr.Lo,F
	skpnc
	incf	char.ptr.Hi,F
	
	movlw	LOW char.base.addr	; add index to character table
	addwf	char.ptr.Lo,F		; program memory start address
	skpnc
	incf	char.ptr.Hi,F
	movlw	HIGH char.base.addr
	addwf	char.ptr.Hi,F

	movlw	.5
	movwf	inst.data
	movlw	.4
	btfsc	voxel.func, Fcharyz
	movwf	vox.y
	btfss	voxel.func, Fcharyz
	clrf	vox.z
	
inst.charyz.loop
	call	position.xyz
	call	get.char.data
	clrf	PCLATH
	
	btfsc	voxel.func, Fvoxel.set
	iorwf	INDF,F
	btfsc	voxel.func, Fvoxel.inv
	xorwf	INDF,F
	
	xorlw	0xFF
	btfsc	voxel.func, Fvoxel.clr
	andwf	INDF,F
	
	incf	char.ptr.Lo,F
	skpnz
	incf	char.ptr.Hi,F
	
	btfsc	voxel.func, Fcharyz
	decf	vox.y,F
	btfss	voxel.func, Fcharyz
	incf	vox.z,F
	
	decfsz	inst.data,F
	goto	inst.charyz.loop
		
	call	restore.xyz
	goto	inst.load

get.char.data
	movfw	char.ptr.Hi
	movwf	PCLATH
	movfw	char.ptr.Lo
	movwf	PCL
	
	
;------------------------------------------
; unpack X,Y,Z values
;
; X,Y,Z values are packed as shown
; [1  0  0   c  c  c  c x2 ]  instruction
; [x1 x0 y2 y1 y0 z2 z1 z0 ]  inst.data
	
unpack.xyz	movfw	inst.data
	movwf	vox.y
	andlw	0x07
	movwf	vox.z

	rrf	vox.y,F
	rrf	vox.y,F
	rrf	vox.y,W
	andlw	0x07
	movwf	vox.y

	movfw	inst.data
	movwf	vox.x
	rrf	instruction,W
	rlf	vox.x,F
	rlf	vox.x,F
	rlf	vox.x,W
	andlw	0x07
	movwf	vox.x
	return
;------------------------------------------
; Unpack register R0-R7 and setup FSR
; strip instruction out

unpack.register
	movfw	instruction
	andlw	0x07
	addlw	cp.regbase
	movwf	FSR
	
	rrf	instruction,F
	rrf	instruction,F
	rrf	instruction,W
	andlw	0x07
	movwf	instruction
	return

;------------------------------------------
; Instruction Pointer Return Address Stack

push.address	
	movfw         addr.stack.pointer
              movwf         FSR
              movfw	inst.ptr.Lo
              movwf         INDF
	incf	FSR,F
	movfw	inst.ptr.Hi
	movwf	INDF
	incf	FSR,W
	movwf	addr.stack.pointer
	return


pull.address	decf	addr.stack.pointer,W
	movwf	FSR
	movfw	INDF
              movwf	inst.ptr.Hi
	decf	FSR,F
	movfw	INDF
	movwf	inst.ptr.Lo
	movfw	FSR
	movwf	addr.stack.pointer
	return
		
;------------------------------------------
; Stack	
inst.pushr    movfw         INDF
              call          stack.push
              goto          inst.load

inst.pullr	movfw	FSR		; save FSR which has already been setup to point
	movwf	inst.data		; to the register we want to pull the stack back to.

	call          stack.pull		; pull data from stack
              movwf         stack.data.temp	; save data in temp variable

              movfw	inst.data		; load saved FSR value
              movwf	FSR		; and write back to FSR so we point at register

              movfw         stack.data.temp             ; read temp variable back into W
              movwf         INDF                        ; and write pulled data back into register via INDF
				; Thus demonstating the shortcomings of having only
				; one index register and one working register.

       	goto          inst.load
              
              
inst.pushxyz  call          stack.pushxyz
              goto          inst.load
              
inst.pullxyz  call          stack.pullxyz
              goto          inst.load              
              
              
;------------------------------------------
stack.push    movwf         stack.data.temp
              movfw         user.stack.pointer
              movwf         FSR
              movfw         stack.data.temp
              movwf         INDF

              incf          user.stack.pointer,f
	movfw	user.stack.pointer
	xorlw	(user.stack+.33+.1)
	movlw	user.stack
	skpnz
	movwf	user.stack.pointer
              return

stack.pull    movfw	user.stack.pointer
	xorlw	user.stack
	movlw	(user.stack+.33)
	skpz
	decf          user.stack.pointer,W
              movwf         user.stack.pointer

              movwf         FSR
              movfw         INDF
              return
;----------------------------------------------
	; ( a b --- b )
inst.drop	call	stack.pull
	goto	inst.load

	; (a b --- b a )
inst.swap	call	stack.pull
	movwf	savex	; use for temp store of top stack value
	call	stack.pull
	movwf	savey	; use for temp store of top stack-1 value
	movfw	savex
	call	stack.push
	movfw	savey
	call	stack.push
	goto	inst.load

	;( a --- a a)
inst.dup	call	stack.pull
	call	stack.push
	movfw	stack.data.temp
	call	stack.push
	goto	inst.load

	; (a b --- a b a )
inst.over	call	stack.pull
	movwf	savex	; use for temp store of top stack value
	call	stack.pull
	movwf	savey	; use for temp store of top stack-1 value
	movfw	savex	
	call	stack.push
	movfw	savey
	call	stack.push
	movfw	savex
	call	stack.push
	goto	inst.load


	; (a b c --- c a b  )
inst.rot	call	stack.pull
	movwf	savex	; use for temp store of top stack value
	call	stack.pull
	movwf	savey	; use for temp store of top stack-1 value
	call	stack.pull
	movwf	savez	; use for temp store of top stack-2 value	

	movfw	savey
	call	stack.push
	movfw	savex
	call	stack.push
	movfw	savez
	call	stack.push
	goto	inst.load

inst.tstz	call	stack.pull
	call	zero.flag
	call	stack.push
	goto 	inst.load

;----------------------------------------------

stack.pushxyz movfw         vox.x
              call          stack.push
              movfw         vox.y
              call          stack.push
              movfw         vox.z
              call          stack.push
              return


stack.pullxyz call          stack.pull
              movwf         vox.z
              call          stack.pull
              movwf         vox.y
              call          stack.pull
              movwf         vox.x
              return

;------------------------------------------
; logic operations
inst.xor	call	stack.pull
	movwf	math.op1
	call	stack.pull
	xorwf	math.op1,W
	call	zero.flag
	call	stack.push
	goto	inst.load


inst.add	call	stack.pull
	movwf	math.op1
	call	stack.pull
	addwf	math.op1,W
	call	carry.flag
	call	stack.push
	goto	inst.load

inst.sub	call	stack.pull
	movwf	math.op1
	call	stack.pull
	subwf	math.op1,W
	call	carry.flag
	call	stack.push
	goto	inst.load


inst.and	call	stack.pull
	movwf	math.op1
	call	stack.pull
	andwf	math.op1,W
	call	zero.flag
	call	stack.push
	goto	inst.load


inst.or	call	stack.pull
	movwf	math.op1
	call	stack.pull
	iorwf	math.op1,W
	call	zero.flag
	call	stack.push
	goto	inst.load

inst.not	call	stack.pull
	xorlw	0xFF
	call	zero.flag
	call	stack.push
	goto	inst.load


carry.flag	bcf	flag.bit, Fcarry
	skpnc
	bsf	flag.bit, Fcarry
	; drop into zero.flag since we need both tests
zero.flag	bcf	flag.bit, Fzero
	skpnz
	bsf	flag.bit, Fzero
	return




;------------------------------------------
; Command fetcher
; get instruction and post increment inst.ptr.

inst.fetch	movfw	inst.ptr.Hi
	movwf	PCLATH
	
	movfw	inst.ptr.Lo
	incf	inst.ptr.Lo,F
	skpnz
	incf	inst.ptr.Hi,F
	movwf	PCL

;------------------------------------------
; Instruction command decode fault control
;
; If an instruction fails to decode correctly we end up 
; here. Used during development, we should no longer
; get here with the production code so in the unlikely event 
; it does the code halts.

fault.command1
fault.command2
	goto	$


;-------------------------------------------	
; Random number generator

rand.num      movwf   mult.mult2         ; put W into mult2

; -----------------------------< START >-----------------------------------------------------                            
rand.gen      ; Mark Jeronimus of Digital Mosular shares this code: (from www.piclist.com)
              ; 16-bit random generator (tested)
              ; The 0xA1 is part of a prime polynom value 0xA1A1 which generates a full LFSR

              clrc
              rrf           randomH,F
              rrf           randomL,F
              skpc
              goto          rand.done
              movlw         0xA1
              xorwf         randomH,F
              xorwf         randomL,F
;
; -----------------------------< END >------------------------------------------------------

rand.done	; check if random number is 0x000, if so reseed it to a non-zero value
	movfw         randomH
              iorwf         randomL,W
              skpnz
              call          rand.reseed

	; sum the high and low bytes of the random number to get an 8 bit result, ignore carry
	; use this as the random number in the 8 bit multiply that follows
	movfw	randomL
	addwf	randomH,W

                            
              ; The drawing processor LDRND instruction generates an integer value between 0 and the command argument-1
	; To do this we multiply the random number by the value passed as the LDRAND argument and use the high byte
	; of the result.  The code is an 8x8 multiply with a 16 bit result.
	
	clrf    mult.resHi         ; initialise resHi, resLo
              clrf    mult.resLo
              bsf     mult.resLo,7       ; set bit 7 in resLo to use as loop counter
                         
rand.num.loop
              rrf     mult.mult2,F       ; rotate mult2 one bit right
              skpnc                      ; test bit shifted out of mult2
              addwf   mult.resHi,F       ; if it was a 1, add W to ResHi
              rrf     mult.resHi,F       ; shift 16 bit result right one bit
              rrf     mult.resLo,F       ; 
              skpc                       ; skip next if carry set as we have shifted 8 times
              goto    rand.num.loop      ; if carry was not set, loop again
              movfw   mult.resHi         ; load random number in to W
              return  
              
rand.reseed   movfw         TMR0
              skpnz
              goto          $-2
              movwf         randomL
	xorlw	0xFF
	movwf	randomH
              return


;-----------------------------------------------------
save.xyz	movfw	vox.x
	movwf	savex

	movfw	vox.y
	movwf	savey

	movfw	vox.z
	movwf	savez
	return


restore.xyz	movfw	savex
	movwf	vox.x

	movfw	savey
	movwf	vox.y

	movfw	savez
	movwf	vox.z
	return



;-----------------------------------------------------
; 
; Check that the PIC program code all resides in first 2K program page
; otherwise issues will be caused by incorrect setting of PCLATH when
; call and goto instructions are executed.
;
	if (HIGH $  & b'11000')
	
	 error "*** Program code crosses 2K program page boundary in lc5x5_dp.inc ***"
	 error "   This will cause issues with PCLATH and call / goto instructions   "
	endif
;-----------------------------------------------------
	
