;------------------------------------------------
; 5x5x5 LED cube drawing functions
;
;      Y-axis 0,4,0
;      ^
;     |    / Z-axis 0,0,4
;     |   /
;     |  /
;     | /
;     |/__________>  X-axis
;     0,0,0        4,0,0
;
;
;     Increasing Z-axis values increase depth into cube
;     Z=0 front of cube, Z=4 back of cube.
;
; voxel.op.
;	set - subsequent commands will set voxel
;	clr - subsequent commands will clear voxel
;	inv - subsequent commands will invert voxel
;
;	All other commands operate on the current voxel.op

; voxel	
; 	set/clr/inv voxel position (vox.x, vox.y, vox.z)
;
; voxel.all
;	set/clr/inv all voxels
;
; voxel.line.x
;	set/clr/inv line on X-axis at vox.y, vox.z 
;
; voxel.line.y
;	set/clr/inv line on Y-axis at vox.x, vox.z 
;
; voxel.line.z
;	set/clr/inv line on Z-axis at vox.x, vox.y 
;
; buffer.shift
;	commands shift entire buffer up, down, left, right
;	forwards or backwards.  
;
; draw.line
;	Draw line starting from current X,Y,Z co-ordinates.
;	Arguments required are XYZ increment/decrement
;	and line length.
;
;
;----------------------------------------------------------------
;
; LED Matrix draw functions
;
; Linear display memory buffer to LED mapping
;                    
; disp.buffer.4	4 | 21, 22, 23, 24, 24  
; disp.buffer.3	3 | 16  17, 18, 19, 20  
; disp.buffer.2	2 Z 11, 12, 13, 14, 15  
; disp.buffer.1	1 |  6,  7,  8,  9, 10  
; disp.buffer.0	0 |  1,  2,  3,  4,  5  
;                              <------ X -------->  
; 	                  0   1   2   3   4      Y = 0

; disp.buffer.9	4 | 21, 22, 23, 24, 24  
; disp.buffer.8	3 | 16  17, 18, 19, 20  
; disp.buffer.7	2 Z 11, 12, 13, 14, 15  
; disp.buffer.6	1 |  6,  7,  8,  9, 10  
; disp.buffer.5	0 |  1,  2,  3,  4,  5  
;                              <------ X -------->  
; 	                  0   1   2   3   4      Y = 1
;etc....to Y=4

; disp.buffer bit field to LED map
;  7   6   5   4   3   2   1   0 <- X bit
;  -----------------------------
;  x   x   x   21  22  23  24  25   	Buffer n+4
;
;  x   x   x   16  17  18  19  20  	Buffer n+3
;
;  x   x   x   11  12  13  14  15 	Buffer n+2
;
;  x   x   x    6   7   8   9  10 	Buffer n+1
;
;  x   x   x    1   2   3   4   5 	Buffer n
;
;-----------------------------------------------------------------------------------------

;Shift register mapping  (shift registers are STP16CP05 16 bit shift register )

;            Shift Reg #1 output                 SOUT -> SIN     Shift Reg #2 output
;  1  2  3  4  5   6  7  8  9 10   11 12 13 14 15   16 |  1 2 3 4   5 6 7 8 9 
; ---------------+---------------+----------------+---------------+--------------------------
; 25 24 23 22 21  20 19 18 17 16   15 14 13 12 11   10 |  9 8 7 6   5 4 3 2 1
;                               LED #



; Function Sets / Clears / Inverts voxel at vox.x, vox.y, vox.z

voxel	btfsc	voxel.func, Fvoxel.set
	goto	voxel.set
	btfsc	voxel.func, Fvoxel.clr
	goto	voxel.clr
	btfss	voxel.func, Fvoxel.inv
	return

voxel.inv	call	position.xyz
	xorwf	INDF,F
	return

voxel.set	call	position.xyz
	iorwf	INDF,F
	return	

voxel.clr	call	position.xyz
	xorlw	0xFF
	andwf	INDF,F
	return	

;----------------------------------------------------------------------------------------------
; Function sets FSR pointing at YZ position in buffer.draw and returns with W == X bit position

position.xyz
	; vox.y x 5 to get layer
	movfw	vox.y
	movwf	FSR
	clrc
	rlf	FSR,F
	rlf	FSR,F
	addwf	FSR,F	

	;add result to drawing buffer memory base address
	movlw	buffer.draw
	addwf	FSR,F
	
	; then add vox.z to set buffer offset
	movfw	vox.z
	addwf	FSR,F

	; return with W = X bit position 
x.bit.pos	movlw	HIGH x.lookup
	movwf	PCLATH
	movfw	vox.x
	addwf	PCL,F
x.lookup	retlw	1<<4
	retlw	1<<3
	retlw	1<<2
	retlw	1<<1
	retlw	1<<0


;-----------------------------------------------
; Whole buffer functions
; Clear, Set and Invert
;
voxel.all.inv	movlw	.25
	movwf	voxel.count
	movlw	buffer.draw
	movwf	FSR
	comf	INDF,F
	incf	FSR,F	
	decfsz	voxel.count,F
	goto	$-3
	return

; Need for speed, so we do these with in-line code rather than a loop.
voxel.all.set	movlw	b'00011111'
voxel.all.clr	movwf	buffer.draw
	movwf	buffer.draw+.1
	movwf	buffer.draw+.2
	movwf	buffer.draw+.3
	movwf	buffer.draw+.4
	movwf	buffer.draw+.5
	movwf	buffer.draw+.6
	movwf	buffer.draw+.7
	movwf	buffer.draw+.8
	movwf	buffer.draw+.9
	movwf	buffer.draw+.10
	movwf	buffer.draw+.11
	movwf	buffer.draw+.12
	movwf	buffer.draw+.13
	movwf	buffer.draw+.14
	movwf	buffer.draw+.15
	movwf	buffer.draw+.16
	movwf	buffer.draw+.17
	movwf	buffer.draw+.18
	movwf	buffer.draw+.19
	movwf	buffer.draw+.20
	movwf	buffer.draw+.21
	movwf	buffer.draw+.22
	movwf	buffer.draw+.23
	movwf	buffer.draw+.24
	return


;-----------------------------------------------
; Line drawing functions
; 

voxel.line.x	movlw	b'00011111'
	btfsc	voxel.func, Fvoxel.set
	movwf	INDF

	btfsc	voxel.func, Fvoxel.clr
	clrf	INDF

	btfsc	voxel.func, Fvoxel.inv
	xorwf	INDF,F
	return

voxel.line.y	call	voxel
	call	voxel.bit.modY
	call	voxel.bit.modY
	call	voxel.bit.modY
	call	voxel.bit.modY

	return

voxel.line.z	
	call	voxel
	call	voxel.bit.modZ
	call	voxel.bit.modZ
	call	voxel.bit.modZ
	call	voxel.bit.modZ
	return

voxel.bit.modY
	incf	FSR,F
	incf	FSR,F
	incf	FSR,F
	incf	FSR,F
voxel.bit.modZ
	incf	FSR,F
	btfsc	voxel.func, Fvoxel.set
	iorwf	INDF,F

	btfsc	voxel.func,Fvoxel.inv
	xorwf	INDF,F

	btfsc	voxel.func,Fvoxel.clr
	andwf	INDF,F

	return




;-----------------------------------------------
; Plane drawing functions

voxel.plane.x 
	clrf	vox.z
	clrf	vox.y
	call	voxel.line.y
	incf	vox.z,F
	call	voxel.line.y
	incf	vox.z,F
	call	voxel.line.y
	incf	vox.z,F
	call	voxel.line.y
	incf	vox.z,F
	call	voxel.line.y

	return

voxel.plane.y	clrf	vox.z
	call	position.xyz
	call	voxel.line.x
	incf	FSR,F
	call	voxel.line.x
	incf	FSR,F
	call	voxel.line.x
	incf	FSR,F
	call	voxel.line.x
	incf	FSR,F
	call	voxel.line.x
	return

voxel.plane.z	clrf	vox.y
	call	position.xyz
	call	voxel.line.x
	movlw	.5
	addwf	FSR,F	
	call	voxel.line.x
	movlw	.5
	addwf	FSR,F	
	call	voxel.line.x
	movlw	.5
	addwf	FSR,F	
	call	voxel.line.x
	movlw	.5
	addwf	FSR,F	
	call	voxel.line.x

	return



buffer.shift.y.up
	movlw	buffer.draw +.19
	movwf	stack.data.temp
buffer.shift.y.loop.up
	movfw	stack.data.temp
	movwf	FSR
	movfw	INDF
	incf	FSR,F
	incf	FSR,F
	incf	FSR,F
	incf	FSR,F
	incf	FSR,F
	movwf	INDF
	decf	stack.data.temp,F
	movfw	stack.data.temp
	xorlw	buffer.draw - .1
	skpz
	goto	buffer.shift.y.loop.up
	clrf	buffer.draw
	clrf	buffer.draw+.1
	clrf	buffer.draw+.2
	clrf	buffer.draw+.3
	clrf	buffer.draw+.4
	return


buffer.shift.y.down
	movlw	buffer.draw+.5
	movwf	stack.data.temp
buffer.shift.y.loop.down
	movfw	stack.data.temp
	movwf	FSR
	movfw	INDF
	decf	FSR,F
	decf	FSR,F
	decf	FSR,F
	decf	FSR,F
	decf	FSR,F
	movwf	INDF
	incf	stack.data.temp,F
	movfw	stack.data.temp
	xorlw	buffer.draw+.25
	skpz
	goto	buffer.shift.y.loop.down
	clrf	buffer.draw+.20
	clrf	buffer.draw+.21
	clrf	buffer.draw+.22
	clrf	buffer.draw+.23
	clrf	buffer.draw+.24
	return



buffer.shift.z.in
	movlw	buffer.draw +.23
	movwf	stack.data.temp
buffer.shift.z.loop.in	
	movfw	stack.data.temp
	movwf	FSR
	movfw	INDF
	incf	FSR,F
	movwf	INDF
	decf	stack.data.temp,F
	movfw	stack.data.temp
	xorlw	buffer.draw - .1
	skpz
	goto	buffer.shift.z.loop.in
	clrf	buffer.draw
	clrf	buffer.draw+.5
	clrf	buffer.draw+.10
	clrf	buffer.draw+.15
	clrf	buffer.draw+.20
	return
	
buffer.shift.z.out
	movlw	buffer.draw+.1
	movwf	stack.data.temp
buffer.shift.z.loop.out	
	movfw	stack.data.temp
	movwf	FSR
	movfw	INDF
	decf	FSR,F
	movwf	INDF
	incf	stack.data.temp,F
	movfw	stack.data.temp
	xorlw	buffer.draw +.25
	skpz
	goto	buffer.shift.z.loop.out
	clrf	buffer.draw+.4
	clrf	buffer.draw+.9
	clrf	buffer.draw+.14
	clrf	buffer.draw+.19
	clrf	buffer.draw+.24
	return

buffer.shift.x.right

	movlw	.25
	movwf	stack.data.temp
	movlw	buffer.draw
	movwf	FSR

buffer.shift.x.right.loop	
	bcf	INDF,5
	rrf	INDF,F
	incf	FSR,F
	decfsz	stack.data.temp,F
	goto 	buffer.shift.x.right.loop
	return
	

buffer.shift.x.left

	movlw	.25
	movwf	stack.data.temp
	movlw	buffer.draw
	movwf	FSR

buffer.shift.x.left.loop	
	clrc	
	rlf	INDF,F
	incf	FSR,F
	decfsz	stack.data.temp,F
	goto 	buffer.shift.x.left.loop
	return


;---------------------------------------------------
; draw line
;
; instruction
; 7 6 5 4 3 2 1 0
;               |_ X dec
;
; inst.data
; 7 6 5 4 3 2 1 0
; | | | | | | | |_
; | | | | | | |___ } Line length [ 1 - 5 ]
; | | | | | |_____ }
; | | | | |_______ X inc
; | | | |_________ Y inc
; | | |___________ Y dec
; | |_____________ Z inc
; |_______________ Z dec


draw.line	

line.loop	call	voxel		; modify voxel at vox.x, vox.y, vox.z

	decf	inst.data,F		; decrement line length and exit when 0
	movlw	0x07
	andwf	inst.data,W
	skpnz
	goto	line.exit	

	;-- X --			
	btfss	inst.data,3		
	goto	$+5		; Leave X unchanged
	incf	vox.x,W		; increment X 
	btfsc	vox.x,2
	clrw
	movwf	vox.x

	btfss	instruction,0		; Leave X unchanged
	goto	$+6		; decrement X
	movlw	.4
	movf	vox.x,F
	skpz
	decf	vox.x,W
	movwf	vox.x



	
	;-- Y --
	btfss	inst.data,4		; Leave Y unchanged
	goto	$+5		; increment Y
	incf	vox.y,W
	btfsc	vox.y,2
	clrw
	movwf	vox.y


	btfss	inst.data,5		; Leave Y unchanged
	goto	$+6		; decrement Y
	movlw	.4
	movf	vox.y,F
	skpz
	decf	vox.y,W
	movwf	vox.y


	;-- Z --
	btfss	inst.data,6		; leave Z unchanged
	goto	$+5		; increment Z
	incf	vox.z,W
	btfsc	vox.z,2
	clrw
	movwf	vox.z

	btfss	inst.data,7		; leave Z unchanged
	goto	$+6		; decrement Z
	movlw	.4
	movf	vox.z,F
	skpz
	decf	vox.z,W
	movwf	vox.z

	goto	line.loop		; repeat

line.exit	;call	stack.pullxyz		; restore X.Y.Z
	return			; we're done

	
	
; rotate x axis through cube centre line( y=2, z=2 )
; 
rotate.x	
	movfw	buffer.draw
	movwf	rotate
	movfw	buffer.draw+.20
	movwf	buffer.draw
	movfw	buffer.draw+.24
	movwf	buffer.draw+.20
	movfw	buffer.draw+.4
	movwf	buffer.draw+.24
	movfw	rotate
	movwf	buffer.draw+.4

	movfw	buffer.draw+.1
	movwf	rotate
	movfw	buffer.draw+.15
	movwf	buffer.draw+.1
	movfw	buffer.draw+.23
	movwf	buffer.draw+.15
	movfw	buffer.draw+.9
	movwf	buffer.draw+.23
	movfw	rotate
	movwf	buffer.draw+.9


	movfw	buffer.draw+.2
	movwf	rotate
	movfw	buffer.draw+.10
	movwf	buffer.draw+.2
	movfw	buffer.draw+.22
	movwf	buffer.draw+.10
	movfw	buffer.draw+.14
	movwf	buffer.draw+.22
	movfw	rotate
	movwf	buffer.draw+.14

	movfw	buffer.draw+.3
	movwf	rotate
	movfw	buffer.draw+.5
	movwf	buffer.draw+.3
	movfw	buffer.draw+.21
	movwf	buffer.draw+.5
	movfw	buffer.draw+.19
	movwf	buffer.draw+.21
	movfw	rotate
	movwf	buffer.draw+.19

	movfw	buffer.draw+.6
	movwf	rotate
	movfw	buffer.draw+.16
	movwf	buffer.draw+.6
	movfw	buffer.draw+.18
	movwf	buffer.draw+.16
	movfw	buffer.draw+.8
	movwf	buffer.draw+.18
	movfw	rotate
	movwf	buffer.draw+.8

	movfw	buffer.draw+.7
	movwf	rotate
	movfw	buffer.draw+.11
	movwf	buffer.draw+.7
	movfw	buffer.draw+.17
	movwf	buffer.draw+.11
	movfw	buffer.draw+.13
	movwf	buffer.draw+.17
	movfw	rotate
	movwf	buffer.draw+.13


	return
