; ********************************************************************************
; RGB PWM LED driver with serial comms interface
; Internet Release.  Code version 1.0.0 (28/07/2006)
; Pete Griffiths (c) 2006
; Written and assembled using MPLAB v7.31
;
; ********************************************************************************
; Note: For 12F629 and 12F675
; All timings are based on the internal 4Mhz PIC oscillator.  The OSCCAL value is read
; from program memory at 0x3FF and written to the OSCAL register.  This value must be
; present and correct. 
; a) If it is not present the code will crash.
; b) If it's incorrect the bit rate for the serial comms may be too far out to receive
;    data reliably if at all.
;
; This does not apply to 12F683 as it doesn't use this method to calibrate the oscillator
;
; ********************************************************************************
; Ensure that GPIO3 is either driven or pulled up but NOT LEFT FLOATING
;                       ___ ___
;                  Vdd -|1 ^ 8|- Vss
;                  Rxd -|2   7|- red
;  unused. tie to  Vdd -|3   6|- green
;  unused. tie to  Vdd -|4   5|- blue
;	        
; ********************************************************************************
;
;
;
; ********************************************************************************
; MPLAB listing config

  LIST b=16, n=0, t=ON, st=On, c=80
; b  - tab spacing
; n  - lines per page
; c  - column width
; t  - truncate lines on, wrap lines off
; st - print symbol table on/off

  IFDEF  __12F683
	#include "p12f683.inc"
	#define ADCpresent
  ENDIF

          
  IFDEF  __12F675
	#include "p12f675.inc"
	#define ADCpresent
  ENDIF

  IFDEF  __12F629
	#include "p12f629.inc"
	
  ENDIF
	errorlevel -302	; suppress banksel warning messages
  IFDEF __12F683	
     __CONFIG       _CP_OFF & _WDT_OFF & _BOD_OFF &_PWRTE_ON & _INTRC_OSC_NOCLKOUT & _MCLRE_OFF & _CPD_OFF
  ELSE
     __CONFIG       _CP_OFF & _WDT_OFF & _BODEN_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT & _MCLRE_OFF & _CPD_OFF
  ENDIF
       
; Program application GPR variables
; Note: GPR address space for 12F629/675 starts 0x20,
; ends at 0x5F and is not banked.
; --------------------------------------------
;          
	cblock 0x20 
	 start		;20	RxD byte 0 (packet start marker 0x81) 
	 rdata	   	;21	RxD byte 2 (red data)
	 gdata		;22	RxD byte 3 (green data)
	 bdata		;23	RxD byte 4 (blue data)
 	 checksum		;24	RxD byte 5 (checksum)
	 validate		;25	running checksum of data received in each packet
	 bitTime		;26	counter used in the bit time delay function
	 pwmramp		;27	running 'ramp' counter used by PWM function
	 rled		;28	red   data, transferred to rpwm when pwmramp == 0
	 gled		;2A	green data, transferred to gpwm when pwmramp == 0
	 bled		;2B	blue  data, transferred to bpwm when pwmramp == 0
	 rpwm		;2C	red   LED PWM drive level [0=O%, 255=99%]
	 gpwm		;2D	green LED PWM drive level [0=O%, 255=99%]
	 bpwm		;2E	blue  LED PWM drive level [0=O%, 255=99%]

	endc


;        
; Define program constants
; -----------------------------------------
cSTARTDELAY	equ	.6

	           
; GPIO port definitions
; -----------------------------------------
; Internal weak pull-ups need to be configured manually in the 'Startup and initialisations" 
; section as needed.

rxd	EQU	.5	;in	Serial Received Data
unused4	EQU	.4	;in	
unused3	EQU	.3	;in  
blue	EQU	.2	;out 	Blue LED drive out
green	EQU	.1	;out	Green LED drive out
red	EQU	.0	;out 	Red LED drive out


; Bank Select pseudo instructions
#define	setbank0	bcf	STATUS,RP0	; Sel Bank 0
#define	setbank1	bsf	STATUS,RP0	; Sel Bank 1


; ******************************************************************************************************

	; Reset Vector
	; ------------
	org	0x000
_resetVector	goto	_startup

; ******************************************************************************************************
; Main code block
; Receive serial data and construct and validate 5 byte packets
; Drive RGB LED PWM outputs
;
; Asynchronous serial data is received on RxD input at 2400 bits per second
; Frames use 1 start, 8 data , 1 stop and no parity
; The software is written to receive a fixed length packet of 5 bytes
; The first frame is a start-of-packet delimiter, always 0x81
; The last byte is a checksum of the values in the previous four frames
; The checksum is computed by the sender to make validation as fast as possible for the receiver.
; When the received checksum is added to the sum of all four preceeding frames, the result should
; be zero. If not the packet is discarded.
; 
; Packet format. 5 frames
; 0: start      = 0x81 start delimiter, always 0x81
; 1: red        = 0x00-0xFF RED PWM value  	
; 2: green      = 0x00-0xFF GREEN PWM value	
; 3: blue       = 0x00-0xFF BLUE PWM value  	
; 4: checksum	= 2's complement of sum [frames 0::3]
;
;       T01234567P
;     |_XXXXXXXX    Serial data frame starT D0-D7 stoP 
;     (Note: RS-232 inverts the levels: 0 = +3 to +15V, 1= -3 to -15V)
;
; ------------------------------------------------------------------------------------------------------

_newPacket	clrf 	validate	; clear packet checksum validation byte
	movlw	start	; get memory location of start variable (first byte of packet)
	movwf 	FSR	; write to FSR (indirection register)

_startBit	movlw	cSTARTDELAY	; use frame byte as temporary counter
	movwf	INDF	;
_waitStart	call	_pwmDrive	; 
	btfsc	GPIO,rxd	; wait for start bit (0-20us from falling edge)
	goto	_waitStart	; 
	decfsz	INDF,F	; decrement temporary counter
	goto	_waitStart	; test start bit cSTARTDLEAY times consecutively with 24uS interval
			; should now be in middle of start bit +/- 24uS
			; depending on when it was detected (bit length is 416uS)
			
			; frame byte is used as counter for the start bit code
			; so it always contains zero when we get here.
	bsf	INDF,7	; set bit 7 (used to detect 8th bit shifted in)

_nextBit	call	_waitBit	; wait full bit period to see middle(ish) of RxData bit
	rrf	INDF,F	; rotate right RxData byte and catch bit in Carry
	btfsc	GPIO,rxd	; test RxD line (serial bit)
	bsf	INDF,7	; if high set bit 7 of RxData byte
	skpc		; test carry to see if '1' rotated out
	goto	_nextBit	; if not wait for next bit
	
_stopBit	call	_waitBit	; wait 1 bit time
	btfss	GPIO,rxd	; test for 1 - is a stop bit
	goto	_newPacket	; if not 1, framing error, discard packet 
	
	movfw	start	; compare first byte received 
	xorlw	0x81	; with 0x81 (start-of-packet delimiter)
	bnz	_newPacket	; if not 0x81, restart packet receive

	movfw	INDF	; put last byte received in W
	addwf	validate,F	; add to validate variable and save 
	
	movfw	FSR	; compare FSR (indirection address pointer) with GPR memory...
	addlw	-(checksum)	; address of last byte (checksum) by subtraction
	bz	_validatePkt	; if they're the same (result 0), we received 6 bytes
	incf	FSR,F	; otherwise increment FSR to point to next byte
	goto	_startBit	; and get the next frame
	
_validatePkt	movf	validate,F	; validate has been summing all received frames, including checksum
	bz	_goodPacket	; if contents are now zero, the validation is good
	goto	_newPacket	; otherwise packet is bad, Disgard packet
	
			; --------------------------------------------------------------------------
					; 
_goodPacket	movfw	rdata	; put received Red data into W
	movwf	rled	; save W in Red LED latch 
	movfw	gdata	; same as Red code but for Green
	movwf	gled
	movfw	bdata	; same as Red code but for Blue
	movwf	bled
	goto	_newPacket



; ******************************************************************************************************
; Bit Time delay
; 2400bps = 416uS per bit
; ------------------------------------------------------------------------------------------------------

_waitBit	movlw	.17	;3
	movwf	bitTime	;4

_bitLoop	call	_pwmDrive	;6
	decfsz	bitTime,F	;21
	goto	_bitLoop	;23   repeat bit loop
	clrc	 	; must return with Carry bit clear

	return		;3   
; ******************************************************************************************************
; PWM code
; from call to return = 20uS
; compares current RG & B led values with 8 bit (256 step) pwmramp and activates GPIO output accordingly
; note: LED will never be 100% on. 
;       With xpwm=255 led will be off for 1 pwm period
;       or minimum mark/space ratio is 0%, maximum is 99%
;
; LED data is only transferred to the LED PWM registers when the PWM Ramp is at zero.
; This adds 6uS to a PWM Drive cycle once in every 256 calls to the pwmDrive function.

; ------------------------------------------------------------------------------------------------------
;			;Time uSeconds
_pwmDrive	incf	pwmramp,F	;3 (call is 2 + incf = 3)
	skpz		;4 
	goto	_pwmEntry	;6

_transfer	clrf	GPIO	;
	movfw	rled	;  
	movwf	rpwm	;  
	movfw	gled	;  
	movwf	gpwm	;  
	movfw	bled	;  
	movwf	bpwm	;  

_pwmEntry	movfw	pwmramp	;7 
	addwf	bpwm,W	;8 
	skpnc		;9
	bsf	GPIO,blue	;10   
	movfw	pwmramp	;11
	addwf	gpwm,W	;12
	skpnc		;13
	bsf	GPIO,green	;14
	movfw	pwmramp	;15
	addwf	rpwm,W	;16
	skpnc		;17
	bsf	GPIO,red	;18
	return		;20 (return is 2 instruction cycles)
	


; ******************************************************************************************************
; Initialisation and startup code block
; ------------------------------------------------------------------------------------------------------

_startup	clrf	GPIO	; clear GPIO output before setting TRIS register
	setbank1		; switch to register bank 1
	
  IFDEF OSCCAL			; defined only for 12F629 / 675
	call 	0x3FF	; read factory oscillator calibration value
	movwf	OSCCAL	; write to OSCCAL register
			; If using a 12F629/675 this value must be present
			; and correct, or the code will not function properly
			; if it functions at all.
  ENDIF

  IFDEF ADCpresent
	clrf	ANSEL	; Set ports for digital mode (12F675 / 12F683 only)
  ENDIF
	movlw	~(1<<red | 1<<green | 1<<blue ) 
	movwf	TRISIO	; Sets TRIS register
	
	; Set weak-pull up (enable as required here)
	; ---------------------------------------------------------------
	bcf	OPTION_REG,NOT_GPPU ; enable weak-pull up
	bsf	WPU,5	; enable weak-pull-up on GPIO5
			; GPIO3 has no weak-pull-up feature
  	setbank0
	movlw	0x07	; load W=7
  IFDEF __12F683
	movwf	CMCON0	; disable Comparator
  ELSE
	movwf	CMCON	; disable Comparator
  ENDIF

	; Clear GPR memory
	; ---------------------------------------------------------------
	
	movlw	0x20	; load W with address of first GPR memory
	movwf	FSR	; save in FSR indirection register
	movwf	0x5F	; save a non-zero value to last GPR memory
_clrNext	clrf	INDF	; clear contents of register pointed to by FSR
	incf	FSR,F	; increment FSR
	movf	0x5F,F	; Test value in last GPR memory location
	skpz		; skip next if last GPR has been cleared
	goto	_clrNext	; otherwise, repeat until done.
	

	goto	_newPacket

;	------------------------------------------------------------------------
	; Embed code revision into program memory
	dt "ssrgbdrv.asm 28/07/2006 "  
	dt "(c) Pete Griffiths 2006 "
	dt "mailto:ssrgbdrv@petesworld.demon.co.uk" 
	dt "free to use in non-commercial applications only"
;	------------------------------------------------------------------------
;	fill 00, 0x3FF-$
; *************************************************************************************************************

			
; *************************************************************************************************************			

	end

