; ------------------------------------------------------------------------------------------
; Interrupt driven asynchronous serial packet transmitter
;
; This codes sends 1 packet consisting of 5 frames.
; The packet format is:
; 1 Start packet delimiter. Always 0x81
; 2	Data byte 1
; 3	Data byte 2
; 4	Data byte 3
; 5	Checksum
;
; Checksum = 2's compliment of the sum of Frames 1 to 4
;
; A Frame is 1 Start Bit, 8 Data bits, 1 Stop bit.
; 1 = High, 0 = Low. 
; (Note: RS232 uses inverted logic levels:  1 = -12V, 0 = +12V)
;	
	movwf	save_W	; Preserve W, STATUS and PCLATH
	swapf	STATUS,w
	movwf	save_Status
	movfw	PCLATH
	movwf	save_PCLATH
	banksel	TMR0	; set bank 0 while in interrupt handler

	bsf	GPIO,bitclock	; set bit clock output high

	movlw	cBITCLOCK	; reload timer 0 
	movwf	TMR0	; for 416uS serial bit clock
	bcf	INTCON,T0IF	; clear timer 0 interrupt flag
	
	movf	intCount,F	; test interupt counter
	skpz		; skip next if counter 0
	decf	intCount,F	; else decrement counter by 1
	
	bcf	GPIO,bitclock	; set bit clock output low
			
	call	_sendPacket	; call send packet state machine
	call	_TxByte	; call transmit byte state machine
	call	_sendChanges
	

	

	movfw	save_PCLATH	;Restore PCLATH
	movwf	PCLATH
	swapf	save_Status,W	;Restore STATUS and W
	movwf	STATUS	
	swapf	save_W,F
	swapf	save_W,W

	retfie

;---------------------------------------------------------------------------------------
; Serial packet transmission 
;---------------------------------------------------------------------------------------

_sendPacket	movf	pktSent,F	
	skpz		; if pktSent = 0, still sending packet
	return		; if pktSent = 1 finished sending packet
	
	movf	TxdSent,F
	skpnz		; if TxdSent = 1 ready for next frame
	return		; if TxdSent = 0 still sending current frame
	
	clrf	TxdSent	
	clrf	PCLATH
	movfw	packetState
	addwf	PCL,F
	goto	_pktStart
	goto	_pktData
	goto	_pktChksum
		
_pktStart	movlw	0x81
	movwf	chksum
	movwf	txByte
	movlw	data1
	movwf	FSR
	incf	packetState,F
	return
	
_pktData	movfw	INDF
	movwf	txByte	
	addwf	chksum,F
	incf	FSR,F
	movlw	chksum
	xorwf	FSR,W
	skpz
	return
	incf	packetState,F
	return
	
_pktChksum	comf	chksum,F
	incf	chksum,W
	movwf	txByte
	movlw	0xFF
	movwf	pktSent
	clrf	packetState
	return
	
;---------------------------------------------------------------------------------------
; Serial frame bit transmission 
;---------------------------------------------------------------------------------------
_TxByte	movf	TxdSent,F
	skpz
	return	
	clrf	PCLATH
	movfw	serialState
	addwf	PCL,F
	goto	_startBit
	goto	_bitShift
_page0check	goto	_stopBit

	if HIGH _page0check != 0
	 error "Computed GOTO in Serial bit transmission function not in page 0"
	endif

_startBit	bcf	GPIO,Txd
	movlw	.8
	movwf	bitCounter
	incf	serialState,F
	return
	
_bitShift	rrf	txByte,F
	skpnc
	bsf	GPIO,Txd
	skpc	
	bcf	GPIO,Txd
	decfsz	bitCounter,F
	return
	incf	serialState,F
	return

_stopBit	bsf	GPIO,Txd
	comf	TxdSent,F
	clrf	serialState
	return

;---------------------------------------------------------------------------------------
; Trigger packet send when RGB data changes
; or periodically (~1.1S) when data is static
;---------------------------------------------------------------------------------------

_sendChanges	movfw	rpwm
	xorwf	rcmp,W
	bnz	_loadPkt
	
	movfw	gpwm
	xorwf	gcmp,W
	bnz	_loadPkt
	
	movfw	bpwm
	xorwf	bcmp,W
	bnz	_loadPkt
	movf	mustSend,F
	skpz
	return	

_loadPkt	movf	pktSent,F	; if packet is still being sent
	skpnz		; we can't send change until it 
	return		; has completed.
	
	movfw	rpwm	; copy current red value
	movwf	data1	; to packet register variable
	movwf	rcmp	; and change compare vairable
	
	movfw	gpwm	; do green as for red
	movwf	data2
	movwf	gcmp
	
	movfw	bpwm	; do blue as for red
	movwf	data3
	movwf	bcmp
	
	clrf	pktSent	; trigger packet send
	clrf	mustSend	; clear 'mustSend' counter
	return	
	
;---------------------------------------------------------------------------------------
