;********************************************************************************
; I/O controller, with serial command line interface
;
; for PIC 16F627 and 16F628
; Pete Griffiths, March 2005
; 
; Written using MPLAB V7.01
; Assemble with MPASM v03.90.01
;********************************************************************************
;
;
; PORTA is used for outputs
; PORTB is used for inputs however, RB1 and RB2 are used by the PIC USART so aren't available
;
;--------------------------------------------------------------------------------
;
; Code displays the following message after a reset
; 'PIC Serial IO controller ready'
; A '#' is used as the command prompt.
; A '?' is sent to the terminal when any command is either not 
; recognised, or contains invalid or insufficent arguments.
; 
; Commands (All commands are lowercase)
;
; v - display firmware version
; i - display value on input port
; o - display value of outputReg variable (see below)
; nx - set output bit x 
; fx - clear output bit x
; tx - toggle output bit x
;      where x is in the range 1 to 8 or 0 to operate on all bits simultaneously
; sxxxxxxxx/ - set output to bit pattern specified by xxxxxxxx mask
;	where x must be 0 or 1. MSB is leftmost. 
;	All 8 bits must be specified and must terminate with '/'
; c - Continous monitor and display of the input port
; . - Stop continous monitor of the inport port and return to command prompt
;
; m[io] - Toggle port value display format [ i = input, o = output ]
;      displays port value as hex 'HH' or binary 'bbbbbbbb' MSB leftmost
;      At startup the format will be hex.
;
; p[io] - Toggles between displaying the port status only, or printing a text string
;         followed by the port status.
;         e.g. with text "Input status : A5", without "A5"
;         At startup print mode will be port status only, no preceeding text.
;
; w[ed] - Enable or Disable the weak pull-up feature on Port B
;         Weak Pull Up is disabled at startup.
;
; R - Do a software restart (note: Uppercase R)
;
;
; Note: All output commands work on the outputReg memory variable.
;       This is copied to the output port register on completion of each command.
;       The show output port command does not read the port register, it reads
;       and displays the value in the outputReg memory variable.	
;--------------------------------------------------------------------------------
;
; Mask off any unsed input bits to so they return as '0' when 'i' command is used
; You must set this to suit your own requirments.
#define	INPUT_MASK	b'11111000'	; Mask out unused input bits
;
; Set port to use for input and output
; You will need to set the port TRIS registers in the _startup code section
; if you change the ports here

#define	IN_PORT	PORTB	;port to use for input
#define	OUT_PORT	PORTA	;port to use for output
; Notes: For PIC16F627A/628A
; RA5 is input only
; RB1,2 are used by the USART for serial I/O
;--------------------------------------------------------------------------------

	list p=16f627a
	#include "p16f627A.inc"

	__CONFIG       _CP_OFF & _WDT_OFF & _BOREN_OFF & _PWRTE_ON & _INTOSC_OSC_NOCLKOUT  & _MCLRE_OFF & _LVP_OFF
               
	errorlevel -302	; suppress banksel warning messages

#define	len .11		;serial input buffer length

	;memory variable assignments
	cblock 0x20		;Bank 0
		bufidx	;receive string buffer index pointer
		txtidx	;write text index pointer
		outputReg	;copy of I/O output data
		outputTmp	;temp copy of outputReg
		asciiLo 	;ascii hex character
		asciiHi 	;ascii hex character
		binary 	;binary value
		a2byte 	;Hi,Lo Ascii to byte
		returnStatus 	;Return status code from called functions
		modein	;display input ports in ASCII hex or binary
		modeout	;display output port status as ASCII hex or binary
		printin	;display description text with input port status
		printout	;display description text with output port status
		delayI	;software delay inner loop
		delayO	;software delay outer loop
		bufferpos:len	;read line input buffer 
	
	endc
	 
	
; These bank select definitions DO NOT set/clear the RP1 bit
; If code doesn't use banks 2/3 we can save a byte each time we set the bank.
; If you add additional code you may need to include instructions for the RP1 bit
;
#define	bank0	bcf	STATUS,RP0	; Sel Bank 0
#define	bank1	bsf	STATUS,RP0	; Sel Bank 1
;======================================================================================
	org	0x000	;RESET VECTOR
	
		
;	org	0x004	;INTERUPT VECTOR

;=====================================================================================
;Startup initialises registers, peripheral devices and variables
;Run once after reset or cold start
;
_startup	bcf	STATUS,IRP	;clear indirect addressing IRP bit
			;since we will only use banks 0 and 1
	movlw	0x07
	movwf	CMCON	;disable comparator
	
	clrf	PORTA	;clear PortA data latches
	clrf	PORTB	;clear PortB data latches
	

	bank1		;Select register bank 1
	clrf	VRCON	;disable Voltage Reference
	
	clrf	TRISA	;Set PortA I/O for output	
			
	;                      **
	movlw	b'11111011'	;RB2,  RB1 used by USART 
	movwf	TRISB	;set PortB I/O
		
	bsf	OPTION_REG,NOT_RBPU  ;Disable weak pull-up on PortB
	
	bcf	TXSTA,SYNC	;USART Async mode
	bsf	TXSTA,BRGH	;USART BRGH low speed
	movlw	d'25'
	movwf	SPBRG	;set BRG for 9600bps
	
	bsf	TXSTA,TXEN	;enable transmit
	
	bank0		;Select register bank 0
	bsf	RCSTA,SPEN	;enable receive
	bsf	RCSTA,CREN	;enable continous receive
			
;===============================================================================
; Start of main code
;
; IO Control loop
; Accepts I/O port control commands from serial interface 
; 
	call	_convDelay
_IOcontrol	movlw	low _textStartup;send Startup message to terminal
	call	_writeText
	clrf	outputReg	;initialise outputReg
	clrf	modein	;initialise display in port mode
	clrf	modeout	;initialise display out port mode
	clrf	printin
	clrf	printout
_IOloop	movlw	low _textPrompt	;send CLI prompt
	call	_writeText
	call	_readLine	;get input line
	call	_cli	;send to CLI 
	goto	_IOloop	;go round again


;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
; 16F62xA serial I/O routines
; _serialTX	transmit byte in W reg, W preserved
; _serialRX	receives byte and returns in W
	#include 	"sio_txrx_16F62x.inc"
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
; Functions to set output bits
; Code not portable. Used in SerialTerm and TempTerm
	#include	"io_out_bits.inc"
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
; Ascii / Hex conversions
; functions
; _h2a	Converts byte to two character ASCII hex
; _a2h	Converts single ASCII character to binary value
; _a2byte	Converts two ASCII characters to binary byte
	#include	"asciihex.inc"
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
	#include	"delay.inc"
;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

;===============================================================================		
; Build input line
; Buffer length is defined at top of this asm file
; Input character is written to buffer and echoed back to terminal
; 0x00 is end-of-data delimiter.
; If input data exceeds buffer length, a 0 delimter is returned in the first buffer character
; note: terminating [CR] is not echoed back to terminal

; 
_readLine	movlw	bufferpos	;Base address of input buffer
	movwf	FSR	;write to indirect addressing register
_readChar	call	_serialRX	;read character from Serial RX buffer
	movwf	INDF	;store received character in memory buffer
	xorlw	0x0D	;compare character to 0x0D [Carrige Return]
	skpnz		;skip if input not complete [CR] 
	 goto	_readDone	;else goto to exit code
	movfw	INDF	;reload buffered character into W 
	call	_serialTX	;echo received character back to terminal
	incf	FSR,F	;increment read buffer index
	movfw	FSR	;move index pointer into W
	xorlw           bufferpos + len ;test for buffer_end
	bnz	_readChar	;read next character if buffer not full
	movlw	bufferpos	;else reset buffer index pointer
	movwf	FSR	;back to start of buffer in memory
_readDone	clrf	INDF	;and write end-of-data delimiter to buffer
	return		;return


;====================================================================================	
; Read a text string and transmit to terminal
; string MUST be terminated with 0x00
; Call with required string index Low byte base address in W (e.g movlw  low  _textBase)
; Code checks to see if a [CR] was sent and automatically inserts a [LF].
;
_writeText	movwf	txtidx	;put text string base address in index
_wtloop	call	_getText	;get character from string to transmit
	iorlw	0x00	;test for 0x00, end-of-string delimiter
	skpnz		;skip if not end
	 return		;else return
	call	_serialTX	;otherwise call data transmit routine
	xorlw	0x0D	;Test if we just sent a [CR]
	bnz	_writeInc	;skip if we didn't
	movlw	0x0A	;load W with [LF] character
	call	_serialTX	;send [LF] to terminal
_writeInc	incf	txtidx,F	;increment index to point at next character
	goto	_wtloop	;go round again
	
	
;**************************************************************************************
;Command Line Interface.
;Test first character in input buffer
;if it's 0 return
;If match is found, do the command
;else send a '?' to terminal and return
;
;As long as commands find characters pertinent to their function they will ignore any
;following garbage in the input buffer. The command will work correctly regardless. 
;- given program memory constaints it was considered unnecessary to check the whole buffer.
;
_cli	movlw	bufferpos	;put buffer base in W
	movwf	FSR	;set FSR register
	movfw	INDF	;read first char' from buffer
	bnz	_cliSelect	;if !=0 do command select code
	return		;else buffer empty, exit subroutine

_cliSelect	 movlw	'n'	;Test for 'n'
	 xorwf	INDF,W
	bz	_onCode	;Do set output bit(s)
	
	 movlw	'f'	;Test for 'f'
	 xorwf	INDF,W
	bz	_offCode	;Do clear output bit(s)
	
	 movlw	't'	;Test for 't'
	 xorwf	INDF,W
	bz	_toggleCode	;Do toggle output bit(s)
	
	 movlw	'v'	;Test for 'v'
	 xorwf	INDF,W	
	bz	_versionCode	;Do send code version text string
	
	 movlw	'i'	;Test for 'i'
	 xorwf	INDF,W
	bz	_inputCode	;Do send current input status
	
	 movlw	'c'	;Test for 'c'
	 xorwf	INDF,W
	bz	_inputMon	;Do continous monitor input status
	
	 movlw	'o'	;Test for 'o'
	 xorwf	INDF,W
	bz	_outStatCode	;Do send current output status
	
 	 movlw	'm'	;Test for 'm'
	 xorwf	INDF,W
	bz	_modeCode	;Do toggle port status format mode
	
 	 movlw	'p'	;Test for 'pm'
	 xorwf	INDF,W
	bz	_printCode	;Do toggle port status desctiption

	movlw	'w'	;Test for 'w'
	 xorwf	INDF,W
	bz	_wpuCode	;weak pull-up Port B enable
	
	 movlw	's'	;Test for 's'
	 xorwf	INDF,W
	bz	_byteSetCode	;Do byte set output register
	
	movlw	'R'	;Test for 'R'
	 xorwf	INDF,W
	bz	_startup	;soft reset PIC
	

			;If we got here, no valid command
			;in the input buffer
_badCLI	movlw	low _textQuestion ;load text string pointer
	call	_writeText	;and send to terminal
	return		;exit subroutine.

;******************************************************************	
;Send S/W version to terminal
_versionCode	movlw	low _textVersion
	call	_writeText
	return

;******************************************************************	
;Sets Binary or ASCII hex mode independantly for Input and Output ports
;
_modeCode	incf	FSR,F	;increment input buffer
	movlw	'i'	;load W with i
	xorwf	INDF,W	;test
	bnz	_moden1	;skip not i
	 comf	modein,F	;else invert all bits of modein
	return
_moden1	movlw	'o'	;load W with o
	xorwf	INDF,W	;test
	bnz	_badCLI	;skip not o
	 comf	modeout,F	;else invert all bits of modeout
	return
;******************************************************************	
;Sets printing of descriptive text string for Input and Output ports
;
_printCode	incf	FSR,F	;increment input buffer
	movlw	'i'	;load W with i
	xorwf	INDF,W	;test
	bnz	_printl	;skip not i
	 comf	printin,F	;else invert all bits of printin
	return
_printl	movlw	'o'	;load W with o
	xorwf	INDF,W	;test
	bnz	_badCLI	;do badCLI if not o
	 comf	printout,F	;else invert all bits of printout
	return
	
;******************************************************************	
; Enable / Disable Weak Pull-Up feature on Port B
; At startup and after using the 'R' command, weak-pull-up is disabled.
;
_wpuCode	incf	FSR,F	;increment input buffer
	movlw	'e'	;load W with e
	xorwf	INDF,W	;test
	bnz	_wpul	;skip not e
   	 bank1
   	 bcf	OPTION_REG,NOT_RBPU  ;Enable weak pull-up on PortB
   	 bank0
	return
_wpul	movlw	'd'	;load W with d
	xorwf	INDF,W	;test
	bnz	_badCLI	;do badCLI if not d
	 bank1
	 bsf	OPTION_REG,NOT_RBPU  ;Disbale weak pull-up on PortB
	 bank0
	return

;******************************************************************	
;Read OutputReg memory and send value back to terminal 
_outStatCode	movlw	low _textNewline;load text string pointer
	call	_writeText	;send text to terminal
	movlw	low _textOutput	;load text string pointer
	btfsc	printout,0	;test print status and skip if 0
	call	_writeText	;send text to terminal
	movfw	outputReg	;Read I/O Port
	btfsc	modeout,0	;test value in var mode
	 goto	_writeBin	;if !=0 do  binary display	
_writeAscii	call	_h2a	;Convert to ASCII hex
	movfw	asciiHi	;load ASCII hex hi nibble in W
	call	_serialTX	;send to terminal
	movfw	asciiLo	;load ASCII hex lo nibble in W
	call	_serialTX	;send to terminal	
	return		;all done.	
	
;******************************************************************
;Read PortB and send value back to terminal 
_inputCode	movlw	low _textNewline;load text string pointer
	call	_writeText	;send text to terminal
_inputCodeNCR	movlw	low _textInput	;load text string pointer
	btfsc	printin,0	;Test print status and skip if 0
	call	_writeText	;send text to terminal
	movfw	IN_PORT	;Read I/O Port
	andlw	INPUT_MASK	;Mask of output bits 
	btfsc	modein,0	;test value in var mode
	 goto	_writeBin	;if !=0 do  binary display	
	goto	_writeAscii	;else use the writeAscii code in
			;outputReg display
;******************************************************************
; Input monitor
; Monitor input and write to terminal until character received
;  	
_inputMon	movlw	low _textNewline;load text string pointer
	call	_writeText	;send text to terminal
_inputMonL	movlw	0x00	;load W for 1mS delay
_inputWait	addlw	0x01	;add 1 to W
	bnz	_inputWait	;repeat until W = 0
	call	_inputCodeNCR	; display input
	movlw	0x0D	;set CR to display (no LF)
	call	_serialTX	;send to terminal
	btfss	PIR1,RCIF	;test RX flag status
	 goto	_inputMonL	;monitor inputs until RX data received

 	call	_serialRX	;Get RX data
 	xorlw	'.'	;is at a '.'?
	skpz		;skip out if it is
 	goto	_inputMonL 	;else continue to monitor inputs
 	return 
	
	
;*******************************************************************************
;Write binary ASCII string to terminal
;Enter with binary byte in W
;No return status
;var binary is destroyed
_writeBin	movwf	binary	;Save W in var binary
	setc		;set Carry
	rlf	binary,F	;do first shift
_binLoop	movlw	'0'	;load W with ASCII char' 0
	 skpnc		;skip next if Carry clear		;
	  movlw	'1'	;else load W with ASCII' 1
	 call	_serialTX	;send to terminal
	 clrc		;clear Carry
	 rlf	binary,F	;rotate binary left through carry
 	 movf	binary,f	;test var binary 
	bnz	_binLoop	;go round again if binary != 0
	return		;done. return to call.

;===============================================================================
;Computed goto's read text strings		
;
_getText	movlw	high _textPrompt;put high byte of _getText address in W
	movwf	PCLATH	;and write to PCLATH
	movf	txtidx,w	;load W with index value
	movwf	PCL	;and put in PCL
			;jump into data table which contains
			;character string as RETLW instructions

	org	0x300	;Put lookup table at a page boundary

	if high _textPrompt != high _textEnd
	  messg " ATTENTION: routine '_getText' is crossing a page boundary "
	endif
	; /r = [CR] character. _writeText routine will automatically insert a [LF}
	; when it sends a [CR]			
_textPrompt	dt	"\r# ",0	
_textQuestion	dt	"\r? ",0
_textInput	dt	"Input status : ",0
_textOutput	dt	"Output status: ",0
_textStartup	dt	"\rPIC Serial IO controller ready\r\n",0
_textNewline	dt	"\r",0
_textVersion	dt	"\rFW V2.1 20/03/2005\r",0
_textEnd	dt	0
	
	
	
	end
