; ********************************************
; Electronic Die for PIC 16F690
; Pete Griffiths,  30 March 2007
; email: pic@petesworld.demon.co.uk
; Copyright Pete Griffiths 2007
;
; Use PIC type 16F690
;                   __ __
; [ICSP 5V ] Vdd  -|1  20|-  Vss  [ICSP Ov ] 
;            n/c  -|2  19|-  n/c  [ICSP CLK ] 
; [ICSP Vpp] n/c  -|3  18|-  n/c  [ICSP DAT ]
;            n/c  -|4  17|-  SW1
;            n/c  -|5  16|-  n/c
;           LEDA  -|6  15|-  n/c
;           LEDB  -|7  14|-  n/c
;           LEDC  -|8  13|-  n/c
;           LEDD  -|9  12|-  n/c
;            n/c  -|10 11|-  n/c
;                   -----
; n/c -> no connection
; [ ICSP connections ] if you want to program the PIC using ICSP
;
; LED Arrangment for die display
;  B   C      
;  D A D
;  C   B
; ********************************************
;
;
;
  #include "p16f690.inc"	
  
  __CONFIG       _CP_OFF & _WDT_OFF & _BOR_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT  & _MCLRE_ON & _FCMEN_OFF & _IESO_OFF & _CPD_OFF
           
  errorlevel -302	; suppress banksel warning messages


; ********************************************
;           Equates and Variables
  
            cblock 0x20
              dtime         ;variable used in delay sub
              ltime         ;variable used in delay sub
              rollCnt       ;pseudo roll counter
              dieNum        ;pseudo roll die display
              throw         ;actual die throw number
              blankto       ;blank time out counter
            endc



; PORTB is unused, all pins configured as input with weak-pull-up enabled
; 
; Throw switch SW1 must be on PORTA, 
; All PORTA pins configured with weak-pull-up enabled
SW1	equ	2      ; Throw switch Port A I/O line

; LED Arrangment for die display
; B   C      
; D A D
; C   B
; LEDs must connect or PORTC
LEDA	set	(1<<4) ; Set port bit for each LED
LEDB	set	(1<<3)
LEDC	set	(1<<7)
LEDD	set	(1<<6)
LEDSOFF	equ	(LEDA | LEDB | LEDC | LEDD)

; ********************************************
; Point reset vector to program start
;
              org           0x00
	goto 	_startup
;
; Embed code version and copyright notice.
              dt            "Pete Griffiths (c) R1.2.31032007. Electronic Die for 16F690"
;
; ********************************************          
; Entry point for program from Reset
; Set INTCON register for interupt on PORTA/B int-on-change
; disable Global interupts so it wakes from sleep
; but doesn't service interupt.
; Enable weak pull up on PORTA/B input.
; test LEDs and leave off
; Initialise required peripherals and registers at startup
;
      
_startup	clrf	PCLATH	; set PCLATH to 0 ready for lookup table in main code
	movlw	LEDSOFF	; 
	movwf	PORTC	; set Port C outputs high (turn off LEDs)
	
	bsf	STATUS,RP0	; select register bank 1

	movlw	b'00000000'	;setup option register
	;	  ||||||||---- PS0 - Timer 0: 
	;	  |||||||----- PS1
	;	  ||||||------ PS2
	;	  |||||------- PSA -  Assign prescaler to Timer0
	;	  ||||-------- TOSE - LtoH edge
	;	  |||--------- TOCS - Timer0 uses IntClk
	;	  ||---------- INTEDG - falling edge RB0
	;	  |----------- NOT_RABPU - pull-ups enabled
	movwf	OPTION_REG
	


	bcf	STATUS,RP0	; This instruction and next
	bsf	STATUS,RP1	; select register bank 2

	clrf	ANSEL	; set analog select to digital
	clrf	ANSELH	; set analog select to digital
	
	movlw	0xFF
	movwf	WPUB	; Enable weak pull-ups on port B

	bcf	STATUS,RP1	; This instruction selects register bank 0
	bsf	STATUS,RP0	; select register bank 1

	movlw	0xFF
	movwf	WPUA	; Enable weak pull-ups on port A
	
	movlw	b'11101111'	
	movwf	TRISA	; set port A 
	clrf	TRISC	; set port C to output
	movlw	0xFF	
	movwf	TRISB	; set port B to input

                bsf   	IOCA, SW1	; enable Int-On-Change RA0
	
	bcf	STATUS,RP0	; select register bank 0

	movlw	b'00001001'     ; Set up INTCON register for
 	movwf	INTCON          ; Port A interrupt on change
	movfw	PORTA           ; Int Flag only
	bcf	INTCON,RABIF    ; Ensure RABIF Int Flag cleared
                clrf            dieNum	; preset program variables
	clrf	throw

	; This little bit of code is a visual indication that the 
	; PIC has initialised and shows all LEDs are working
	movlw	~(LEDA | LEDB)
	movwf	PORTC
       	movlw	.100	; set up 100mS delay 
              	call	_Delay	; 
	movlw	~(LEDA | LEDD)
	movwf	PORTC
       	movlw	.100	; set up 100mS wait 
              	call	_Delay	; 
	movlw	~(LEDA | LEDC)
	movwf	PORTC
       	movlw	.100	; set up 10mS wait 
              	call	_Delay	
	movlw	LEDSOFF
	movwf	PORTC	; turn all LEDs when done


; ********************************************
; put PIC to sleep and wake on SW1 pressed
; then 'throw' die
; PORTA uses internal weak pull up with SW1 open, grounded with SW1 closed
; Uses interupt on change to wake PIC
;
_sleepsw1     movfw         PORTA         ; Read PORTA clears Int-On-Change condition
              bcf           INTCON,RABIF  ; and clear RABIF flag in interrupt register.
          
              sleep                       ; Go to sleep
              nop                         ; 
          
              movfw         PORTA         ; Read PORTA register
              bcf           INTCON,RABIF  ; and clear RABIF flag in interrupt register.
              movlw         d'5'          ; load W with 5 (5mS)
              call          _Delay        ; and call delay to debounce switch down.
          

; ********************************************
; Once PIC wakes from sleep the throw counter
; runs while SW1 is held down
;
; Turn off LEDs while SW1 is down
;

_throwdie      movlw        LEDSOFF       ; Turn off LEDs while counting
               movwf        PORTC
               clrf         throw         ; Reset throw to 0 for each new throw.

; Number of instruction cycles per loop for this block is 8. 
; Block will loop 125,000 times per second with 4Mhz clock (1uS) instruction cycle
; or 1250 times in 10mS. Since the throw is determined by the length of time the SW1 switch
; is held down, a person pressing the button will result in a random throw.

_loop         incf          throw,F       ; add 1 to throw                         
              movfw         throw         ; get throw and put in W    
              xorlw         .6            ; compare to 6                         
              skpnz         	          ; skip if result < 6             
              clrf          throw         ; else set throw back to 0     
_sw1down      btfss         PORTA,SW1     ; Is SW1 down?                            
              goto          _loop         ; keep counting if it is    
                                          ; we don't debounce the swich because
                                          ; any bounce just helps to make it
                                          ; more random
        

; ********************************************
; switch released
; call rolling subroutine
; then display the thrown number on the LEDs
;
              call          _roller       ; Switch has been released so call rolling die sub
              addlw         -.71
              skpz
              goto          $-1	
              movfw          throw        ; Finished rolling it so get throw value
              call          _lookup       ; call display data lookup
              movwf         PORTC         ; and display the throw on the LEDs
              
; ********************************************
; display thrown number for 20 seconds
; then blank display to save battery
; and go back to sleep awaiting next 'throw'
; checks for SW1 throw while displaying previous throw
; in which case it throws again without sleeping.
;

              movlw         0xD0          ; Set count for 20 second display timeout
              movwf         blankto       ; move to working file register
_blankdly     movlw         .1            ; call delay subroutine
              call          _LDelay       ; with W = 100mS
               
              btfss         PORTA,SW1     ; on return check if SW1 has been pressed
              goto          _throwdie     ; if it has, goto the main throw code

              decfsz        blankto,F     ; otherwise decrement display timeout
              goto          _blankdly     ; and carry on with the timeout count


; the fadeout section of code causes the LEDs to fade down at the end of the
; 20 second timeout period.  This uses a single count value for the fadeout period
; and this is used to control the on/off duty cycle of the LEDs as they fade down
; so the period remains constant but the on/off ratio goes from 1/1 to 1/99

_fadeout      movlw         0xFE          ; Set length of fadeout period > = longer
              movwf         blankto       ; but don't set to 0xFF 
              
              
_fading       movf          throw,w       ; get last throw number
              call          _lookup       ; find LED data for throw number
              movwf         PORTC          ; display throw
              movf          blankto,W     ; get value in blankto, put in W
              call          _fadelay      ; and call fade delay subroutine

              movlw         LEDSOFF       ; set data for all LEDs off in W
              movwf         PORTC         ; write to LEDS
              comf          blankto,W     ; complement blankto and put in W
              call          _fadelay      ; and call fade delay routine

              decfsz        blankto,F     ; -1 from blankto
              goto          _fading       ; keep fading until it reaches 0
              goto          _sleepsw1     ; goto sleep.
              
                            
; *********************************************
; Subroutine
; used by fadeout to give slow fade of LEDs before
; sleeping

_fadelay      movwf         dtime         ; Enter with W set to outer loop count so save this
_fouter       btfss         PORTA,SW1     ; check if SW1 has been pressed
              goto          _throwdie     ; if it has, goto the main throw code
              movlw         0x16          ; Inner loop count
              movwf         ltime         
_finner       decfsz        ltime,F       
              goto          _finner
              decfsz        dtime,F       
              goto          _fouter
              return
; ********************************************
; SUBROUTINE
; Lookup Table for Port Display Data
; Lookup table takes value in W and returns with
; port data for die in W
; Output low = LED on, high = LED off 
;
; B   C      
; D A D
; C   B
;
_lookup       addwf         PCL,F         ;Add W to PCL for computed goto.
                                                                 ; 
              retlw         ~LEDA	      ;die 1
              retlw         ~LEDC	      ;die 2
              retlw         ~(LEDA | LEDB)	      ;die 3
              retlw         ~(LEDB | LEDC)	      ;die 4
              retlw         ~(LEDA | LEDB | LEDC)     ;die 5
_lastretlw    retlw         ~(LEDB | LEDC | LEDD)     ;die 6

; -=* Last retlw instruction must be in page 0 *=-
	if high  _lastretlw != 0
	  messg " LED map data lookup table end not in page zero
	endif


; ********************************************
; Subroutine
; Accurate delay routines
; Two routines. Both called with required delay period in W
; One has delay of W x 1mS. Other has delay of W x 100mS
; Accurate for 4Mhz clock
;
_Delay        movwf         dtime         ; Call for W x 1mS
__Dcall       call          __1mS
              decfsz        dtime,F
              goto          __Dcall
__DlyEnd      return
_LDelay       movwf         ltime         ; Call for W x 100mS
__Dlcall      movlw         d'100'
              call          _Delay
              decfsz        ltime,F
              goto          __Dlcall
              return
__1mS         movlw         0xC6          
_next         nop
              addlw         0xFF
              btfss         STATUS,Z
              goto          _next
              nop
              nop
              nop
              return

; ********************************************
; SUBROUTINE
; Displays a rolling die thats starts fast and slows to
; a stop about 3 seconds after SW1 released.
;
_roller       movlw         .48
              movwf         rollCnt

_rollerCoast  movfw          dieNum
              call          _lookup
              movwf         PORTC
 	
              movfw         rollCnt
              call          _Delay
             
              movfw         dieNum
              incf          dieNum,F
              xorlw         .6		
              skpnz
              clrf          dieNum		
	
              movlw         .2
              addwf         rollCnt,F
              btfss         rollCnt,7	; loop until rollCnt > 0x80	
              goto          _rollerCoast	; 
_rollerDone   movlw          .6
              movwf         PCL
              return
   
; ********************************************              
              


              end

