; ********************************************************************************
; edge triggered quad delayed outputs
; Production Code 2.0.0 (31/05/2006)
; Pete Griffiths (c) 2006
; Written and assembled using MPLAB v7.21
;
; ********************************************************************************
; Ensure that GPIO3 is either driven or tied but NOT LEFT FLOATING
;                       ___ ___
;                  Vdd -|1 ^ 8|- Vss
;       reset  input -|2   7|- out1
;                 out4 -|3   6|- out2
;  trigger __/ input -|4   5|- out3
;                       
; Outputs go high at timed intervals after a low-to-high transistion of trigger input.
; After the fourth output has gone high, a high level on the reset input will clear
; the outputs and return to wait for another low-to-high transition on the trigger input.
;
; By connecting out4 to the reset input, the code will be self resetting.
;
; ********************************************************************************
; Note: For 12F629 and 12F675
; All timings are based on the internal 4Mhz PIC oscillator.  The OSCCAL value is read
; from program memory at 0x3FF and written to the OSCCAL register.  This value must be
; present and correct. 
; a) If it is not present the code will crash.
; b) If it's incorrect the software delays will not be accurate
;
; This does not apply to the 12F683 which uses a different method.
; 
; ********************************************************************************
;
; MPLAB listing config

  LIST b=16, n=0, t=ON, st=On, c=80
; b  - tab spacing
; n  - lines per page
; c  - column width
; t  - truncate lines on, wrap lines off
; st - print symbol table on/off

  IFDEF  __12F683
                #include "p12f683.inc"
                #define ADCpresent
  ENDIF

          
  IFDEF  __12F675
                #include "p12f675.inc"
                #define ADCpresent
  ENDIF

  IFDEF  __12F629
                #include "p12f629.inc"
                
  ENDIF
                errorlevel -302 ; suppress banksel warning messages
  IFDEF __12F683                
     __CONFIG       _CP_OFF & _WDT_OFF & _BOD_OFF &_PWRTE_ON & _INTRC_OSC_NOCLKOUT & _MCLRE_OFF & _CPD_OFF
  ELSE
     __CONFIG       _CP_OFF & _WDT_OFF & _BODEN_OFF & _PWRTE_ON & _INTRC_OSC_NOCLKOUT & _MCLRE_OFF & _CPD_OFF
  ENDIF
       

                

; Set delay times
; -----------------------------------------------------------------------------
; Either multiples of 1mS or 100mS can be set in the main code.
; Each interval can be set independantly to 1mS or 100mS 
; Currently main code sets all delays for 1mS intervals
; 
;                         .0 <= Delay <=.255
delay1          equ             .5              ;  trigger to out1
delay2          equ             .5              ;  out1 to out2
delay3          equ             .5              ;  out2 to out3
delay4          equ             .25             ;  out3 to out4
;
; Delays are software generated and approximate.  Accuracy depends a lot on the
; correct calibration value being written to the OSCCAL register.
; You may need to test and fine tune the values to get the actual delays required.


; GPIO port definitions
; -----------------------------------------------------------------------------
; The code will automatiaclly set the TRIS register correctly according to the values here.

reset           EQU             5               ;in
out4            EQU             4               ;out            
trigger         EQU             3               ;in  
out3            EQU             2               ;out            
out2            EQU             1               ;out            
out1            EQU             0               ;out            

; Program application variables
; -----------------------------------------------------------------------------
;          
                cblock 0x20 
                 dtime          ; used by delay function
                 ltime          ; used by delay function
                endc

; Bank Select pseudo instructions
; -----------------------------------------------------------------------------
;
#define         setbank0        bcf             STATUS,RP0      ; Sel Bank 0
#define         setbank1        bsf             STATUS,RP0      ; Sel Bank 1


; ******************************************************************************************************

                ; Reset Vector
                ; ------------
                org             0x000
_resetVector    goto            _startup

; ******************************************************************************************************
; Main code block

                ; This waits for a low-to-high transition before triggering delayed outputs
                ; This is a software edge trigger - not level triggered
                ; --------------------------------------------------------------------------
_waitLow        btfsc           GPIO,trigger    ; test trigger input
                goto            _waitLow        ; if input is high wait for it to go low
                nop
_waitHigh       btfss           GPIO,trigger    ; test trigger input
                goto            _waitHigh       ; if input low wait for it to go high

                ; --------------------------------------------------------------------------


                ; First delay between trigger input going high and output 1 going high
                ; --------------------------------------------------------------------------
_outDelay1      movlw           delay1          ; load W with 5 (0 => W <= 255)
                addlw           .0              ; add 0 to test for 0
                skpz                            ; Don't call Delay if W=0 (0ms)
                call            _Delay          ; call 1mS delay function
                ; change _Delay above to _LDelay for delay with 100mS intervals
                
                bsf             GPIO,out1       ; set output 1 high
                ; --------------------------------------------------------------------------    

                ; Second delay between output 1 going high and output 2 going high
                ; --------------------------------------------------------------------------
_outDelay2      movlw           delay2          ; load W with 5 (0 => W <= 255)
                addlw           .0              ; add 0 to test for 0
                skpz                            ; Don't call Delay if W=0 (0ms)
                call            _Delay          ; call 1mS delay function
                ; change _Delay above to _LDelay for delay with 100mS intervals 
                
                bsf             GPIO,out2       ; set output 2 high
                ; --------------------------------------------------------------------------
                
                ; Third delay between output 2 going high and output 3 going high
                ; --------------------------------------------------------------------------
_outDelay3      movlw           delay3          ; load W with 5 (0 => W <= 255)
                addlw           .0              ; add 0 to test for 0
                skpz                            ; Don't call Delay if W=0 (0ms)
                call            _Delay          ; call 1mS delay function
                ; change _Delay above to _LDelay for delay with 100mS intervals
                                
                bsf             GPIO,out3       ; set output 3 high
                ; --------------------------------------------------------------------------
                
                ; Fourth delay between output 3 going high and output 4 going high
                ; --------------------------------------------------------------------------
_outDelay4      movlw           delay4          ; load W with 5 (0 => W <= 255)
                addlw           .0              ; add 0 to test for 0
                skpz                            ; Don't call Delay if W=0 (0ms)
                call            _Delay          ; call 1mS delay function
                ; change _Delay above to _LDelay for delay with 100mS intervals
                
                bsf             GPIO,out4       ; set output 4 high
                ; --------------------------------------------------------------------------
                                                
                ; Level triggered reset.
                ; if reset input is high, code clears outputs and waits for next trigger input
                ; --------------------------------------------------------------------------
_reset          btfss           GPIO,reset      ; load W with 255 (0 => W <= 255)
                goto            _reset
                clrf            GPIO            ; clear all outputs 
                goto            _waitLow        ; go back and wait for next lo-to-hi transition
                                                ; of input.
                ; --------------------------------------------------------------------------    
                
                
                
; ******************************************************************************************
; Delay Function
; Software delay relies on accuracy of PIC internal RC oscillator
; Based on instruction execution time of 1uS this function will provide
; delays of W x 1mS or W x 100mS

_Delay          movwf           dtime           ; Call for W x 1mS
__Dcall         call            __1mS
                decfsz          dtime,F
                goto            __Dcall
__DlyEnd        return
_LDelay         movwf           ltime           ; Call for W x 100mS
__Dlcall        movlw           d'100'
                call            _Delay
                decfsz          ltime,F
                goto            __Dlcall
                return
__1mS           movlw           0xC6            
_next           nop
                addlw           0xFF
                btfss           STATUS,Z
                goto            _next
                nop
                nop
                nop
                return

; ******************************************************************************************************
; Initialisation and startup code block
; ------------------------------------------------------------------------------------------------------

_startup        clrf            GPIO            ; clear GPIO output before setting TRIS register
                setbank1                        ; switch to register bank 1
                
  IFDEF OSCCAL                                  ; defined only for 12F629 / 675
                call            0x3FF           ; read factory oscillator calibration value
                movwf           OSCCAL          ; write to OSCCAL register
                                                ; If using a 12F629/675 this value must be present
                                                ; and correct, or the code will not function properly
                                                ; if it functions at all.
  ENDIF

  IFDEF ADCpresent
                clrf            ANSEL           ; Set ports for digital mode (12F675 / 12F683 only)
  ENDIF
                movlw           ~(1<<out1 | 1<<out2 | 1<<out3 | 1<<out4) 
                movwf           TRISIO          ; Sets TRIS register
                
                ; Set weak-pull up (enable as required here)
                ; ---------------------------------------------------------------
                bcf             OPTION_REG,NOT_GPPU ; enable weak-pull up
                bsf             WPU,5           ; enable weak-pull-up on GPIO5
                ;bsf            WPU,4           ; enable weak-pull-up on GPIO4
                ;bsf            WPU,2           ; enable weak-pull-up on GPIO2
                ;bsf            WPU,1           ; enable weak-pull-up on GPIO1
                ;bsf            WPU,0           ; enable weak-pull-up on GPIO0
                                                ; GPIO3 has no weak-pull-up feature

                setbank0
                movlw           0x07            ; load W=7
  IFDEF __12F683
                movwf           CMCON0          ; disable Comparator
  ELSE
                movwf           CMCON           ; disable Comparator
  ENDIF
                clrf            GPIO            ; clear GPIO


                ; Initialise variables
                ; ---------------------------------------------------------------
                ; not using any:-)
                
                goto            _waitLow        ; device initialised, Start main code.

                                

;               ------------------------------------------------------------------------
                ; Embed code revision into program memory
                dt "delayout200.asm 31/05/2006 "  
                dt "(c) Pete Griffiths 2006 "
                dt "contact@petesworld.demon.co.uk" 
;               ------------------------------------------------------------------------


                end

