;********************************************************************************
; Enhanced 5/2 Central Heating and Hot Water Boiler Programmer
;
; for PIC 16F628A
; Pete Griffiths, April-May 2005
; Version 2.0.2 28-May-2005
; 
; Written using MPLAB V7.01
; Assemble with MPASM v03.90.01
;********************************************************************************
;
;
; u[cChH]	Set Central Heating / Hot Water relay on/off
; m[cChH]	Set Central Heating / Hot Water manual mode on/off
; o 	display heating and hot water status
; ds	send output and manual settings state as Ascii Byte 
; dw	display WDT reset count as BCD Ascii byte
; t	display day and time
; s d hh:mm	set time to day d, hour hh, minutes mm.
;	  seconds are reset to 00
; f[ed]	front panel control e=enabled, d=disabled [locked]
;	when disabled LCD displays 'Locked' on line 2, centre
; p	display program settings
; pnx	erase program entry number n
; pn [hHcCawe] hh:mm
;	   set program number n
;	     c Central heating off. C Central heating on
;	     h Hot water off. H Hot water on
;	     a All days. w Week days only. e Weekend only
;	     Must speciy one day option and at least one output to control
;	     Specify any combination in any order.
; 	    space delimits control from time data
;	     hh:mm time to activate setting
;
; v	send firmware version text to serial terminal
; V	send  'VT100 [ESC] c' command - reset to initial state
;
; *** Commands for development and diagnostics
; requires devFucntions.inc 
; see _devCode in this filep8
;
; w aa dd	write byte dd to I2C address aa
; r aa n	read n bytes from I2C device starting at address aa
; R	soft reset. Executes a GOTO 0x000
; q	Quick set RTC seconds register to 57
; Q	Quick set time and day to Monday 21:59:57
; H	enter infinite loop - force WDT (watch dog timeout)
;
;********************************************************************************
;

	list p=16f628a
	#include "p16f628A.inc"

	__CONFIG       _CP_OFF & _WDT_ON & _BOREN_OFF & _PWRTE_ON & _INTOSC_OSC_NOCLKOUT  & _MCLRE_OFF & _LVP_OFF
               
	errorlevel -302	; suppress banksel warning messages
;********************************************************************************

#define	len .16		;serial input buffer length

	;memory variable assignments
	cblock 0x20		;Bank 0
		bufidx	;serial receive string buffer index pointer
		txtidx	;write text index pointer
		asciiLo 	;conversion routine ascii hex character
		asciiHi 	;conversion routine  ascii hex character
		binary 	;binary value (used in ASCII hex conversion)
		a2byte 	;conversion routine Hi,Lo Ascii to byte
		returnStatus 	;Return status code from called functions
		RelayOut	;flag variable for control relay output
		manual	;flag variable for CH/HW manual mode	
		lock	;flag variable for front panel lock out (chctrl_main.asm)	
		fpTimeComp	;value to rollover time on FP setting (fpset.inc)
		fpProgNo	;front panel setup program entry number
		fpProgCounter	;front panel setup program entry, working counter var
		fpEditLimit	;front panel setup program entry, working limit var
		wdtCount	;Watchdog timeout event count
		xmS	;used by msdelay.inc
		lcdbyte	;used by lcd.inc
		lcdBCD	;used by lcd.inc
		switchNow	;key switch input states this pass 
			;N.B. do not change value outside function
		switchLast	;key switch inputs last pass (only valid within function)
			;outside function values are key down edges detected
	
		bufferSave	;Saved value of buffer index 
		bufferpos:len	;serial input line buffer 
		rtc_minutes
		rtc_hours
		rtc_day
		rtc_minutesPrev
		tc_counter1
		tc_temp1
		mainCount	;Counter used in main loop
	
	endc
			
	cblock 0x70	;0x70-7F map to the same location across all banks
		i2c_device
		i2c_address
		i2c_data
		i2c_shift
		i2c_status
		i2c_counter
		i2c_bytecnt
				
		ByteReadBase:0	;define ByteReadBase and
		seconds	;seconds to same memory address
		minutes	;minutes = ByteReadBase+1
		hours	;hours = ByteReadBase+2
		daycontrol:0	;daycontrol = ByteReadBase+3
		day	;day to same memory address as daycontrol
					
	endc
;	
;--------------------------------------------------------------------------------
; Define relay output control command letters for Serial CLI
; These are referenced in mainVxxx.asm, settingsV2.inc and fpset.inc, fpsetProg.inc
;
R1off	EQU	'c'	;port B6
R1on	EQU	'C'	;port B6
R2off	EQU	'h'	;port B7
R2on	EQU	'H'	;port B7
;--------------------------------------------------------------------------------
;	
; These bank select definitions DO NOT set/clear the RP1 bit
; If code doesn't use banks 2/3 we can save a byte each time we set the bank.
; If you add additional code you may need to include instructions for the RP1 bit
;
#define	bank0	bcf	STATUS,RP0	; Sel Bank 0
#define	bank1	bsf	STATUS,RP0	; Sel Bank 1
;
;================================================================================
	org	0x000	;RESET VECTOR
	goto	_startup
		
;	org	0x004	;INTERUPT VECTOR
;			Interupts not used in this application

;--------------------------------------------------------------------------------
;	Include code
;	textfuncs.inc must be first so it locates code in page Zero
	#include	"textfuncs.inc"  
	#include	"msdelay.inc"
	#include	"switchscan.inc"
	#include 	"serial_IO.inc"
	#include	"asciihex.inc"
	#include	"lcdPutTime.inc"
	#include	"lcd.inc"
	#include	"i2c_driver.inc"
	#include	"TimeControl.inc"
	#include	"settingsV2.inc"
	#include	"keyAction.inc"
	#include	"fpset.inc"
	#include	"fpsetProg.inc"

;--------------------------------------------------------------------------------


;================================================================================
;Initialises PIC internal registers and peripheral devices
;Run once after reset or cold start
;
_startup	bcf	STATUS,IRP	;clear indirect addressing IRP bit
			;since we will only use banks 0 and 1
	movlw	0x07
	movwf	CMCON	;disable comparator
	
	clrf	PORTA	;clear PortA data latches
	clrf	PORTB	;clear PortB data latches
	
	bank1		;Select register bank 1
	clrf	VRCON	;disable Voltage Reference
	
	movlw	b'00110000'
	movwf	TRISA	;Set PortA I/O for output
			;Bits 3-0 LCD Data / Key scan
			;Bits 4 Key scan input 
			;Bits 5 unused
			;Bits 6 LCD RS Control / Key Scan
			;Bit  7 LCD E
	
	movlw	b'00111011'
	movwf	TRISB
			;Bit 0 I2C Data
			;Bit 1 Serial RxD
			;Bit 2 Serial TxD
			;Bit 3 I2C Clock
			;Bit 4 unused
			;Bit 5 unused
			;Bit 6 CH Relay
			;Bit 7 HW Relay
		
	bsf	OPTION_REG,NOT_RBPU  ;Disable weak pull-up on PortB
	

	bsf	OPTION_REG,PS2
	bsf	OPTION_REG,PS2
	bsf	OPTION_REG,PSA

	
	bcf	TXSTA,SYNC	;USART Async mode
	bsf	TXSTA,BRGH	;USART BRGH low speed
	movlw	d'25'
	movwf	SPBRG	;set BRG for 9600bps
	
	bsf	TXSTA,TXEN	;enable transmit
	call	_stopI2C	;initialise I2C bus
	
	
	bank0		;Select register bank 0
	bsf	RCSTA,SPEN	;enable Serial receiver
	bsf	RCSTA,CREN	;enable continous receive
			
;===============================================================================
;Initialises external devices and memory variables
;Run once after reset or cold start

_initOther	btfsc	STATUS,NOT_TO	;test WDT status bit
	goto	_normalRst	
	movlw	0x99
	xorwf	wdtCount,W
	skpz
	incf	wdtCount,W	
	addlw	.6
	skpdc
	addlw	-.6
	movwf	wdtCount
	skpnz
	incf	wdtCount,F
	
	goto	_wdtRst
	
_normalRst	clrf	wdtCount
_wdtRst	movlw	.200	;200mS startup delay
	call	_delay	;to allow everything to settle
	
	movlw	low _textVT100init
	call	_writeText	;reset VT100 terminal 
			;place here so terminal has recovered
			;before further data is sent
		
	call	_lcdInit
	movlw	low _textStartup
	call	_writeLCD
	clrf	switchNow	;clear switch variable
	
	call	_lcdClrLine2	;set LCD to line 2, position 1	
	movlw	_textVersion
	call	_writeLCD	;Display firmware version
	call	_2sDelay	;display result for 2 seconds
	call	_lcdClrLine2	;set LCD to line 2, position 1	
	call	_RTCvalidate	;Validate / initialise RTC	
	call	_2sDelay	;display result for 2 seconds
	call	_lcdClr
	
	movlw	0x3D	;retrieve output settings from NVRAM
	movwf	i2c_address	;after power fail / startup
	movlw	0x01
	call	_i2cRead
	movwf	RelayOut	;write to RelayOut memory var'
	
	decf	i2c_address,F	;after power fail / startup
	movlw	0x01
	call	_i2cRead
	movwf	manual	;write to manual memory var'

	decf	i2c_address,F	;after power fail / startup
	movlw	0x01
	call	_i2cRead
	movwf	lock	;write lock memory var'


; Write the HW- and -CH text to LCD display. When installed in the case the front
; panel artwork around the LCD has this text on it so we don't need this info on 
; the LCD.
;	call	_lcdClr	;clear LCD
;	call	_lcdLine2	;set line 2, Char position 1
;	movlw	low _textHW	;load text pointer in W
;	call	_writeLCD	;write HW-  text to LCD
;	
;	movlw	.13	;load char position with 0 offset
;	call	_lcdLine2W	;set line 2, Char position 14
;	movlw	low _textCH	;load text pointer in W
;	call	_writeLCD	;write -CH text to LCD
	
	
	movlw	bufferpos	;load buffer index into W
	movwf	bufferSave	;initialise bufferSave variable
	movlw	0xFF	;Force minutes to unreal value
	movwf	rtc_minutesPrev;to ensure time control runs at startup.
			;so that output settings are applied to relays.

			;Write start text and prompt to serial I/O
	movlw	low _textStartup
	call	_writeText
	movlw	_textPrompt
	call	_writeText
	
; ===========================================================================
; 
; -=* Main Code loop *=-
; 
_mainLoop
	call	_lcdPutTime	;display time on LCD

	call	_timeControl	;test program control and set relay outputs
	call	_switchScan	;scan switches
	btfss	lock,0	;Test lock flag and skip if Front Panel locked
	call	_keyAction	;perform action on active switches
	
	; at 9600bps the serial interface can receive one chacracter ever 1.04mS
	; This loop ensures that we debounce the key switches without missing any
	; serial data.
	movlw	.15	;setup a 15 cycle loop
	movwf	mainCount	;
	
_mainRXtest	movlw	.1	;
	call	_delay	;wait 1mS
		
		;Serial port receiver is polled with non-blocking code
	btfsc	PIR1,RCIF	;test RX flag status
	call	_serialCLI   	;call CLI code if RX buffer full
	
	decfsz	mainCount,F	;mainCount = mainCount -1
	goto	_mainRXtest   ;loop if not zero
	
	call	_writeOutput	;set control outputs and display on LCD
	goto	_mainLoop	;repeat main loop
	
	
; ============================================================================
; Set physical relay outputs according to values in working memory 
; variable RelayOut.
; Update LCD with output status
;
CH	EQU	6	;Port bit for CH relay
HW	EQU	7	;Port bit for HW relay
			;above values are also used to define
 			;flag bits in manual variable
RelayPort	EQU	PORTB	;Port interface to relays

	
_writeOutput	bank0

	;Display Front Panel Lock status
	movlw	.5
	call	_lcdLine2W	
	movlw	low _textSpace
	btfsc	lock,0
	movlw	low _textLock
	call	_writeLCD

	;Test WDT count and display if non-zero
	
	movfw	wdtCount
	bz	_wdtClear
	movlw	.5
	call	_lcdLine2W
	movlw	low _textLCDwdt
	call	_writeLCD
	movfw	wdtCount
	call	_lcdBCD
	
	;Test I2C status and display error if failed
_wdtClear	movfw	i2c_status
	bz	_I2Cclear
	movlw	.5
	call	_lcdLine2W
	movlw	low _textI2Cfail
	call	_writeLCD


	;Display manual/auto timer status
_I2Cclear	movlw	.0
	call	_lcdLine1W	;
	movlw	low _textLCDman
	btfss	manual,HW
	movlw	low _textLCDprg
	call	_writeLCD
	movlw	.13
	call	_lcdLine1W	
	movlw	low _textLCDman
	btfss	manual,CH
	movlw	low _textLCDprg
	call	_writeLCD
	
	btfss	RelayOut, CH		
	call	_CHoff
	btfsc	RelayOut, CH
	call	_CHon	
	btfss	RelayOut, HW	
	call	_HWoff
	btfsc	RelayOut, HW
	call	_HWon

	;After a power fail restart the lock, manual and relay states
	;are restored from NVRAM. To faciliate this they are saved here

	movlw	0x3B
	movwf	i2c_address
	movfw	lock
	call	_i2cWrite

	incf	i2c_address,F
	movfw	manual
	call	_i2cWrite

	incf	i2c_address,F
	movfw	RelayOut
	call	_i2cWrite
	return
	
_CHoff	bcf	RelayPort, CH	
	movlw	.13
	call	_lcdLine2W
	movlw	low _textOff
	goto	_writeLCD
	
_CHon	bsf	RelayPort, CH	
	movlw	.13
	call	_lcdLine2W
	movlw	low _textOnR
	goto	_writeLCD

_HWoff	bcf	RelayPort, HW	
	movlw	.0
	call	_lcdLine2W
	movlw	low _textOff
	goto	_writeLCD
	
_HWon	bsf	RelayPort, HW	
	movlw	.0
	call	_lcdLine2W
	movlw	low _textOnL
	goto	_writeLCD


	

;===============================================================================		
; Serial CLI
; Build input line and pass to CLI on completion
; Buffer length is defined at top of this asm file
; Input character is written to buffer and echoed back to terminal
; 0x00 is end-of-data delimiter.
; If input data exceeds buffer length, a 0 delimter is returned in the first buffer character
; note: terminating [CR] is not echoed back to terminal
; 
_serialCLI	clrf	mainCount	;clear mainCount and add 1
	incf	mainCount,F	;so on return mainLoop code excutes without delay
	
	movfw	bufferSave	;Get saved buffer index
	movwf	FSR	;write to indirect addressing register
_readChar	call	_serialRX	;read character from Serial RX buffer
	movwf	INDF	;store received character in memory buffer
	xorlw	0xFF	;test for receive error
	skpnz
	return
	movfw	INDF	;reload W with character
	xorlw	0x0D	;compare character to 0x0D [Carrige Return]
	skpnz		;skip if input not complete [CR] 
	 goto	_lineDone	;else goto to exit code
	movfw	INDF	;reload buffered character into W 
	call	_serialTX	;echo received character back to terminal
	incf	FSR,F	;increment read buffer index
	movfw	FSR	;move index pointer into W
	xorlw         bufferpos + len ;test for buffer_end
	bz	_QPrompt	;buffer length exceeded
	movfw	FSR
	movwf	bufferSave	;save buffer index pointer
	return		;not received complete line, return

_QPrompt	movlw	low _textQuestion
	call	_writeText
_newPrompt	movlw	bufferpos	;put buffer index base in W
	movwf	bufferSave	;and put in bufferSave
	movlw	low _textPrompt
	call	_writeText	;issue a CLI prompt to terminal
	return		;done
	
_lineDone	clrf	INDF	;write end-of-data delimiter to buffer
	call	_cli	;Call command interpreter
	goto	_newPrompt

;**************************************************************************************
;Command Line Interface.
;Test first character in input buffer
;if it's 0 return
;If match is found, do the command
;else send a '?' to terminal and return
;
;As long as commands find characters pertinent to their function they will ignore any
;following garbage in the input buffer. The command will work correctly regardless. 
;- given program memory constaints it was considered unnecessary to check the whole buffer.
;
_cli	movlw	bufferpos	;put buffer base in W
	movwf	FSR	;set FSR register
	movfw	INDF	;read first char' from buffer
	bnz	_cliSelect	;if !=0 do command select code
	return		;else buffer empty, exit subroutine

_cliSelect	 movlw	'v'	;Test for 'v'
	 xorwf	INDF,W	
	bz	_versionCode	;Do send code version text string
	
	 movlw	'u'	;Test for 'u'
	 xorwf	INDF,W	
	bz	_setRelay	;user control Relay outputs
	
	 movlw	'm'	;Test for 'm'
	 xorwf	INDF,W	
	bz	_manualMode	;set manual / timed operation
	
	
	 movlw	'o'	;Test for 'o'
	 xorwf	INDF,W	
	bz	_showRelay	;show Relay output status
	
	 movlw	'd'	;Test for 'd'
	 xorwf	INDF,W	
	bz	_dataStatus	;send status byte
	
	 movlw	's'	;Test for 's'
	 xorwf	INDF,W	
	bz	_setTime	;set time

	 movlw	'p'	;Test for 'p'
	 xorwf	INDF,W	
	bz	_setProg	;set Program

	 movlw	't'	;Test for 't'
	 xorwf	INDF,W	
	bz	_showTime	;show clock time
	
	 movlw	'f'	;Test for 'f'
	 xorwf	INDF,W	
	bz	_lockFP	;lockout front panel
	
	
	 movlw	'V'	;Test for 'v'
	 xorwf	INDF,W
	bz	_VT100init	;send VT100 reset to intial state 
			;command to serial interface

	;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~	
	; CLI commands used during development and testing
	; This can be removed from the Production code
;	#include	"DevFunctions.inc"
;_devCode	goto	_DevFunctions
	;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

			;If we got here, no valid command
			;in the input buffer
_badCLI	movlw	low _textQuestion ;load text string pointer
	call	_writeText	;and send to terminal
	return		;exit subroutine.

;******************************************************************	
;Send VT100 terminal reset
_VT100init	movlw	low _textVT100init
	call	_writeText	;send init string to terminal
	movlw	0x50	;
	goto	_delay	;wait 50ms, return from delay

	
;******************************************************************	
;Send S/W version to terminal
_versionCode	call	_writeNewLine
	movlw	low _textVersion
	call	_writeText
	return
	
;******************************************************************		
; Front panel lock out command

_lockFP
	incf	FSR,F		
	movlw	'e'
	xorwf	INDF,W		;enable front panel
	bnz	_lockNxt1
	bcf	lock,0		;clear lock flag
	return

_lockNxt1	movlw	'd'		
	xorwf	INDF,W		;disable front panel
	bnz	_badCLI
	bsf	lock,0		;set lock flag
	return
		
;******************************************************************	
;toggle relay outputs
_setRelay
	incf	FSR,F
	movlw	R1off
	xorwf	INDF,W
	bnz	_relNxt1
	bcf	RelayOut,CH		;clear CH bit
	return

_relNxt1	movlw	R1on
	xorwf	INDF,W
	bnz	_relNxt2
	bsf	RelayOut,CH		;set CH bit
	return

_relNxt2	movlw	R2off
	xorwf	INDF,W
	bnz	_relNxt3
	bcf	RelayOut,HW		;clear HW bit
	return
	
_relNxt3	movlw	R2on
	xorwf	INDF,W
	bnz	_badCLI
	bsf	RelayOut,HW		;clear HW bit
	return

;******************************************************************	
;toggle manual / timed mode
_manualMode	incf	FSR,F
	movlw	R1off
	xorwf	INDF,W
	bnz	_manNxt1
	bcf	manual,CH		;clear CH bit
	return

_manNxt1	movlw	R1on
	xorwf	INDF,W
	bnz	_manNxt2
	bsf	manual,CH		;set CH bit
	return

_manNxt2	movlw	R2off
	xorwf	INDF,W
	bnz	_manNxt3
	bcf	manual,HW		;clear HW bit
	return
	
_manNxt3	movlw	R2on
	xorwf	INDF,W
	bnz	_badCLI
	bsf	manual,HW		;clear HW bit
	return

;******************************************************************	
;show output status
_showRelay
	movlw	low _textHeat
	call	_writeText
	
	movlw	low _textOff	
	btfsc	RelayOut,CH
	movlw	low _textOnR
	call	_writeText
	
	movlw	low _textTimed
	btfsc	manual,CH
	movlw	low _textManual
	call	_writeText
	
		
	movlw	low _textWater
	call	_writeText
	
	movlw	low _textOff	
	btfsc	RelayOut,HW
	movlw	low _textOnR
	call	_writeText
	
	movlw	low _textTimed
	btfsc	manual,HW
	movlw	low _textManual
	call	_writeText

	
	return
	
;******************************************************************	
;send data status		; Lo Nibble		;Hi Nibble
;status byte			; 0 = CH off, HW off	;0 = CH prog, HW prog
; 76543210			; 4 = CH on, HW off	;4 = CH man, HW prog
; ||  ||__ CH on=1, 0=off 		; 8 = CH off, HW on	;8 = CH prog, HW man
; ||  |___ HW on= 1, 0=off		; C = CH on, HW on	;C = CH man, HW man
; ||______ CH manual=1, program=0
; |_______ HW manual=1, program=0
;
_dataStatus	call	_writeNewLine
	incf	FSR,F
	movlw	'w'	;test for 's'
	xorwf	INDF,W
	skpnz
	goto	_dataWdtStat
	
	movlw	's'	;test for 'w'
	xorwf	INDF,W
	skpz
	goto	_badCLI

	
	swapf	RelayOut,W	;get bits 6&7 from RelayOut put in W 3&2
	andlw	0x0C	;keep bits 3+2 only
	movwf	binary	;store here temporarily
	movfw	manual	;get value from manual
	andlw	0xC0	;keep bits 3+2 only
	iorwf	binary,W	;combine with bits from RelayOut
	goto	_writeAscii	;write W as Ascii byte to terminal
	
_dataWdtStat	movfw	wdtCount
	goto	_writeAscii
	
	
;******************************************************************	
; Show Time
;
_showTime	movlw	low _textTime
	call	_writeText
		
	movlw	RTC
	movwf	i2c_device
	movlw	0x00
	movwf	i2c_address
	movlw	0x04	;read 3 bytes
	call	_i2cRead
	

_writeTime	movlw	high _dayTextBeg
	movwf	PCLATH
	movfw	day
	call	_dayText
	call	_writeText

	movlw 	' '
	call	_serialTX
	
	movfw	hours
	call	_writeAscii

	movlw	':'
	call	_serialTX
	movfw	minutes
	call	_writeAscii
	
	movlw	':'
	call	_serialTX
	movfw	seconds
	call	_writeAscii
	return




; ************************************************************
; Validate DS1307 RTC ram
; Test RTC OSC bit, if this is set device will be intialised
; Last two bytes of NVRAM should contain the validation data shown
; below. If this is not present NVRAM device will be initialised
;
; Initialisation clears the RTC OSC bit, configures the RTC control 
; register, clears NVRAM and writes two validationd data bytes.
; 
; Validation data 0x3E =42, 0x3F =BB
;
_RTCvalidate	movlw	0xD0	
	movwf	i2c_device	;set DS1307 H/W address
	clrf	i2c_address	;set I2C register address
	movlw	0x01	;request read one byte
	call	_i2cRead	;call I2C read
	btfsc	seconds,7	;test RTC OSC enable bit, skip if clear
	goto	_ds1307Init	;else initialise RTC
	
	movlw	0x3e	;setup and read two validation bytes
	movwf	i2c_address
	movlw	0x02
	call	_i2cRead
	xorlw	0x42	;test first byte
	bnz	_ds1307Init	;init RTC if not correct
	movfw	ByteReadBase
	xorlw	0xBB	;test second byte
	bnz	_ds1307Init	;init RTC if not correct
	
	movlw	low _textValid	;set text pointer
	goto	_writeLCD	;write to LCD and return from function


_ds1307Init	movlw	0xD0	
	movwf	i2c_device
	movlw	0x00
	movwf	i2c_address
	call	_i2cWrite	;enable DS1307 OSC, set seconds to zero 
	movlw	0x07
	movwf	i2c_address
	movlw	0x00
	call	_i2cWrite	;Set DS1307 control register

_rtcRAMinit
	movlw	0x08	;start of user RAM in DS1307
	movwf	i2c_address
	movlw	.54	;bytes of user RAM in DS1307
	movwf 	tc_temp1
_rtcInitLoop	
	clrw		;clear all user NVRAM in DS1307
	call	_i2cWrite
	incf	i2c_address,F
	decfsz	tc_temp1,F
	goto	_rtcInitLoop	;repeat until complete
	
	movlw	0x3E
	movwf	i2c_address
	movlw	0xBB	
	call	_i2cWrite	;write first validation byte
	incf	i2c_address,F
	movlw	0x42
	call	_i2cWrite	;write second validation byte
	
	movlw	.1	;initialise time and day
	movwf	day	;to Monday 00:00
	clrf	hours
	clrf	minutes
	call	_TandDtoRTC
	

	movlw	low _textRAMinit;set text pointer	
	call	_writeLCD	;use return from fucntion
	goto	_5sDelay
	
	dt	"PIC Heating Programmer / Controller"
	dt	"Copyright (C) 2005. Peter Griffiths"

_clearWDT	;btfss	PORTB,5	;test Port B5, skip next if high.
	clrwdt		;clear watchdog timer	
	return

	end
