;*********************************************************************
; 
; I2C bus software interface for devices with 7-bit H/W addressing.
; Tested with Dallas DS1307 RTC and  Microchip 24LC08 EEPROM
; with 4Mhz PIC Clock, timing supports 100Khz and 400Khz devices
;
; Supports two I2C functions:
; _i2cWrite	Call with data to write in W. Writes one byte
;	Returns status in W. 1=fail, 0=success
;	
; _i2cRead	Call with N number of bytes to read in W. 
;	Returns last byte read in W.  
;	All bytes read saved in ByteReadBase to ByteReadBase + (N-1)
;
;	Note: The maximum number of bytes that can be read at one time
;	maybe dependant on the specific device. The number of bytes that
;	can be stored in memory is dependant on the location of ByteReadBase
;	and other memory use.
;
; For both functions i2c_device and i2c_address variables must be set before
; calling.  These are unaffected by the functions.
;	
; These memory variables must be allocted
; i2c_device		; 7-bit i2c bus address for device
; i2c_address		; memory address
; i2c_data		; byte to be written
; i2c_shift		; used to convert serial data to byte
; i2c_counter		; used to count bits shifted
; i2c_bytecnt		; used to count bytes read
; i2c_status		; call return status: 0x00 = No error. 0x01 = Error
;
;
; This code takes care of all bank switching and exits with bank 0 selected.
;
; On 16F627 / 628 data memory between 0x70 and 0x7F is common RAM and
; maps to the same address in all four banks. Put the variables above
; into this memory to save bank switching in the main code.
;
; Configure the following for the port / bits to use for I2C interface
#define	I2C_PORT  PORTB	;port to use 
#define	I2C_TRIS  TRISB	;tris register to use
#define	SDA 0		;I2C data  port bit
#define	SCL 3		;I2C clock port bit
;
;*********************************************************************

;--------------------------------------------------
;Write byte to I2C device
;Call with data to write in W. W not saved
;i2c_device and i2c_address must be set before calling
;
; On return
; i2c_status = 0, Success
; i2c_status = 1, No Ack received from device
;
_i2cWrite	bank1
	movwf	i2c_data
	call	_clearI2Cbits
	call	_startI2C	;put I2C into start condition
			
	movfw	i2c_device	;load device address + R/W
	andlw	0xFE		;set R/W bit to 0
	call	_writeByte	;send to device
	btfsc	i2c_status,1
	return
	movfw	i2c_address	;load location access address
	call	_writeByte	;send to device
	btfsc	i2c_status,1
	return
	movfw	i2c_data	;load byte to write
	call	_writeByte	;send to device
	btfsc	i2c_status,1
	return
	
	call	_stopI2C	;put I2C bus into stop condition
	bank0		;set register bank to 0
	
	;since the DS1307 uses SRAM there is no internal write cycle delay
	;timing conditions to be met.
	;	
	;Some devices need to recover from a write cycle (EEPROMs).  The delay	
	;here ensures timing for two succesive writes to the device is met.
	;This delay can be tuned according to the specific application / device.
	;
			;If you reusing this code you will need to
;	call	_delay	;provide your own delay routine here as			;
			;required by the device(s) being used.
	movfw	i2c_status	
			
		
	return


;-----------------------------------------------------------------------------
;Read byte from I2C device
; enter with N number of bytes to read in W 
; i2c_device and i2c_address must be set before calling
;
; On return
; last byte read in W
; All bytes read saved to memory locations ByteReadBase to ByteReadBase + (N-1)
; i2c_status = 0, Success
; i2c_status = 1, No Ack received from device
; 
;
_i2cRead	bank1
	movwf	i2c_bytecnt	;save number of bytes to read
	call	_clearI2Cbits ;ensure SDA and SCL bits are clear

	call	_startI2C	;put I2C into start condition
	movfw	i2c_device	;load device address
	andlw	0xFE	;set R/W bit to 0 (Write)
	call	_writeByte	;send to device
	btfsc	i2c_status,1
	return
	movfw	i2c_address	;load location access address
	call	_writeByte	;send to device
	btfsc	i2c_status,1
	return
	
	call	_startI2C	;put I2C bus into start condition
	movfw	i2c_device	;load device address
	iorlw	0x01	;set R/W to 1 (Read)
	call	_writeByte	;send to device
	btfsc	i2c_status,1
	return
	call	_readBytes	;read from device
	call	_stopI2C	;put I2C bus into stop condition

	movfw	INDF	;put last byte read into W
	bank0		;set register bank to 0
	return	

;-----------------------------------------------------------------------------	
; Ensure that output port data latch for I2C bits is 0
_clearI2Cbits	bank0
	movf	I2C_PORT,W	;read port latches
	andlw	~(1<<SCL | 1<<SDA);clear i2c bits, leave others		
	movwf	I2C_PORT	;write back to port output latch
	bank1
	return
;-----------------------------------------------------------------------------
; Enter with data to write in W
; On exit
; SCL low, SDA high
; bank 1 selected
_writeByte	bank1

	movwf	i2c_shift	;save W in shift 
	movlw	0x08	;load W with 8
	movwf	i2c_counter	;save W in bit counter
	call	_holdTime
_writeNxtBit
	rlf	i2c_shift,F	;shift MSB into carry
	skpc		;test carry and skip if 1
	bcf	I2C_TRIS,SDA	;set Data low
	skpnc		;test carry and skip if 0
	bsf	I2C_TRIS,SDA	;set Data high
	call	_clockI2C
	decf	i2c_counter,F	;decrement bit count
	bnz	_writeNxtBit	;do next bit if count !0

	;Test for SDA low - Ack pulse	
	bsf	I2C_TRIS,SDA	;release SDA
	clrf	i2c_status	;clear i2c status
	bsf	I2C_TRIS,SCL	;set Clock high
_AckWait	bank0
	btfss	I2C_PORT,SDA	;test Data line for Ack
	goto	_AckDone	;SDA was low, Ack detected
	bank1		;wait for Ack timed out
	decfsz	i2c_counter,F	;decrement counter
	goto	_AckWait	;wait for Ack

_AckFail	bsf	i2c_status,1 ;set i2c_status to 1 (failure)
	call	_stopI2C	;put stop condition on I2C bus
	bank0		;select bank 0
	movlw	low _textNewline
	call	_writeText
	movlw	low _textI2Cfail
	call	_writeText	
	movlw	low _textNewline
	call	_writeText

	return		;return from here
	
_AckDone	bank1
	call	_holdTime
	bcf	I2C_TRIS,SCL	;set Clock low
	return

;-----------------------------------------------------------------------------
; Read n bytes from I2C slave device 
; On exit
; SCL low, SDA high
; bank 1 selected
_readBytes	movlw	ByteReadBase	;put memory base address in W
	movwf	FSR	;save to indirect file register
_readNxtByt	bsf	I2C_TRIS,SDA;release SDA for slave to control
	movlw	0x08	;load W with 8
	movwf	i2c_counter	;preset counter to read 8 bits
_readNxtBit	clrc		;clear carry
	rlf	INDF,F	;shift input data 1 bit left
	bsf	I2C_TRIS,SCL	;set clock high
	call	_holdTime	;hold clock
	bank0
	movfw	I2C_PORT	;read port into W
	bank1
	bcf	I2C_TRIS,SCL	;set clock low
	andlw	(1<<SDA)	;W contains port data, mask off other bits
	skpz		;skip next if bit was 0
	bsf	INDF,0	;set bit 0 in input data
	decfsz	i2c_counter,F	;decrement counter, skip if done
	goto	_readNxtBit   ;read next bit

	decfsz	i2c_bytecnt,F	;decrement bytes read
	goto	_Ack	;not last byte, do Ack else do Nack

_NAck	bsf	I2C_TRIS,SDA	;release SDA
	call	_clockI2C
	return		;All bytes read. Return

_Ack	bcf	I2C_TRIS,SDA	;pull SDA low
	call	_clockI2C
	incf	FSR,F	;increment index pointer
	goto	_readNxtByt
	
;-----------------------------------------------------------------------------
; Use the return from _clockI2C for the _holdTime delay - saves a byte
; Generate clock
_clockI2C	bsf	I2C_TRIS,SCL	;clock high
	call	_holdTime
	bcf	I2C_TRIS,SCL	;clock low
_holdTime	nop	; call + nop + return = 5uS @ 4Mhz clock
	return	; This will give conservative timing for 100Khz devices

;-----------------------------------------------------------------------------
; Call with bank 1 selected
; On exit
; SCL lo, SDA lo
; bank 1 selected
_startI2C	bsf	I2C_TRIS,SDA	;SDA high
	bsf	I2C_TRIS,SCL	;SCL high
	call	_holdTime
	bcf	I2C_TRIS,SDA	;pull Data low
	call	_holdTime	;wait
	bcf	I2C_TRIS,SCL	;pull Clock low
	call	_holdTime	;wait
	return	
	
;-----------------------------------------------------------------------------	
; Call with bank 1 selected
; On exit
; SCL hi, SDA hi
; bank 1 selected
_stopI2C	bcf	I2C_TRIS,SCL	;ensure Clock is low
	bcf	I2C_TRIS,SDA	;ensure Data is low
	call	_holdTime	;wait
	bsf	I2C_TRIS,SCL	;set Clock high
	call	_holdTime	;wait
	bsf	I2C_TRIS,SDA	;set Data high
	return
		
;-----------------------------------------------------------------------------	
